package org.mule.module.xmlsecurity.keyinfo;

import org.mule.module.xmlsecurity.algorithms.CanonicalizationAlgorithm;
import org.mule.module.xmlsecurity.algorithms.SignatureMethodAlgorithm;

import javax.xml.crypto.dsig.Reference;
import javax.xml.crypto.dsig.SignedInfo;
import javax.xml.crypto.dsig.keyinfo.KeyInfo;
import javax.xml.crypto.dsig.keyinfo.KeyInfoFactory;
import javax.xml.crypto.dsig.keyinfo.X509Data;
import javax.xml.crypto.dsig.spec.C14NMethodParameterSpec;
import java.io.FileInputStream;
import java.security.Key;
import java.security.KeyStore;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import static org.mule.module.xmlsecurity.Signer.fac;

/**
 * <p>Provides the Key Info Using a keystore</p>
 *
 * @author fernando.federico@mulesource.com
 */
public class X509KeyInfoProvider implements  KeyInfoProvider{

    private KeyStore.PrivateKeyEntry keyEntry;
    private String password;
    private String keystoreKey;

    private CanonicalizationAlgorithm canonicalizationAlgorithm;
    private SignatureMethodAlgorithm signatureMethod;

    /**
     * Constructor
     *
     * @param canonicalizationAlgorithm The canonicalization algorithm
     * @param keystore The keystore path
     * @param password The keystore password
     * @param keystoreKey The keystore key
     * @param signatureMethod
     */
    public X509KeyInfoProvider(CanonicalizationAlgorithm canonicalizationAlgorithm, String keystore, String password, String keystoreKey, SignatureMethodAlgorithm signatureMethod) {
        this.canonicalizationAlgorithm = canonicalizationAlgorithm;
        this.password = password;
        this.keystoreKey = keystoreKey;
        this.signatureMethod = signatureMethod;
        try{
            KeyStore ks = KeyStore.getInstance("JKS");
            ks.load(new FileInputStream(keystore), this.password.toCharArray());
            keyEntry = (KeyStore.PrivateKeyEntry) ks.getEntry
                    (keystoreKey, new KeyStore.PasswordProtection(this.password.toCharArray()));
  
        }
        catch (Exception e)
        {
            throw new RuntimeException("Could not read the keystore", e);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public KeyInfo getKeyInfo() {
        X509Certificate cert = (X509Certificate) keyEntry.getCertificate();

        KeyInfoFactory kif = fac.getKeyInfoFactory();
        List x509Content = new ArrayList();
        x509Content.add(cert.getSubjectX500Principal().getName());
        x509Content.add(cert);
        X509Data xd = kif.newX509Data(x509Content);
        return kif.newKeyInfo(Collections.singletonList(xd));
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Key getSecretKey() {
        return keyEntry.getPrivateKey();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public SignedInfo getSignedInfo(Reference ref) {
        try
        {
            return fac.newSignedInfo
                    (fac.newCanonicalizationMethod
                            (canonicalizationAlgorithm.getAlgorithm(),
                                    (C14NMethodParameterSpec) null),
                            fac.newSignatureMethod(signatureMethod.getAlgorithm(), null),
                            Collections.singletonList(ref)); 
        }
        catch(Exception e)
        {
            throw new RuntimeException("Could not create signed info for X509 Signature", e);
        }

    }
}
