package org.mule.metadata.map;

import org.mule.metadata.api.TypeLoader;
import org.mule.metadata.api.model.ArrayType;
import org.mule.metadata.api.model.BooleanType;
import org.mule.metadata.api.model.DateTimeType;
import org.mule.metadata.api.model.DateType;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.NumberType;
import org.mule.metadata.api.model.ObjectFieldType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.model.StringType;
import org.mule.metadata.raml.DataTypeRamlFragmentTypeLoader;

import java.io.File;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;

public class MapTypeLoader implements TypeLoader {

  private static final List<Class<? extends MetadataType>> validSimpleTypeClasses =
      Arrays.asList(StringType.class, NumberType.class, BooleanType.class,
                    DateType.class, DateTimeType.class);

  static {
    System.setProperty("javax.xml.accessExternalDTD", "");
  }

  private final DataTypeRamlFragmentTypeLoader dataTypeRamlFragmentTypeLoader;

  public MapTypeLoader(File file) {
    dataTypeRamlFragmentTypeLoader = new DataTypeRamlFragmentTypeLoader(file, MetadataFormat.JAVA);
  }

  public MapTypeLoader(String content) {
    dataTypeRamlFragmentTypeLoader = new DataTypeRamlFragmentTypeLoader(content, MetadataFormat.JAVA);
  }

  @Override
  public Optional<MetadataType> load(String typeIdentifier) {
    return load(typeIdentifier, null);
  }

  @Override
  public Optional<MetadataType> load(String typeIdentifier, String typeAlias) {
    return dataTypeRamlFragmentTypeLoader.load(typeIdentifier, typeAlias).map(metadataType -> {
      validate(metadataType);
      return metadataType;
    });
  }

  private boolean validArrayType(MetadataType metadataType) {
    if (!(metadataType instanceof ArrayType)) {
      return false;
    }

    ArrayType arrayType = (ArrayType) metadataType;
    final MetadataType elementType = arrayType.getType();

    return validSimpleType(elementType) || validObjectType(elementType);
  }

  private boolean validObjectType(MetadataType metadataType) {
    if (!(metadataType instanceof ObjectType)) {
      return false;
    }

    ObjectType objectType = (ObjectType) metadataType;
    for (ObjectFieldType objectFieldType : objectType.getFields()) {
      if (!(validSimpleType(objectFieldType) || validObjectType(objectFieldType) || validArrayType(objectFieldType))) {
        return false;
      }
    }
    return true;
  }

  private boolean validSimpleType(MetadataType metadataType) {
    for (Class<? extends MetadataType> validSimpleTypeClass : validSimpleTypeClasses) {
      if (validSimpleTypeClass.isAssignableFrom(metadataType.getClass())) {
        return true;
      }
    }
    return false;
  }


  private void validate(MetadataType metadataType) {
    if (validArrayType(metadataType) || validObjectType(metadataType)) {
      throw new RuntimeException(
                                 "Invalid RAML type for Map format.");
    }
  }

}
