package org.mule.datasense.catalog.builder;

import com.google.common.base.Throwables;
import org.mule.datasense.catalog.model.TypesCatalog;
import org.mule.datasense.catalog.util.NotificationMessages;
import org.mule.datasense.common.util.Notifier;

import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;

public class TypesCatalogBuilder {

  private final List<TypesResolverBuilder> typesResolverBuilders;
  private final TypesCatalogBuilderContext typesCatalogBuilderContext;

  private URI baseUri;

  public TypesCatalogBuilder(TypesCatalogBuilderContext typesCatalogBuilderContext) {
    this.typesCatalogBuilderContext = typesCatalogBuilderContext;
    typesResolverBuilders = new ArrayList<>();
  }

  public TypesCatalogBuilder baseUri(URI baseUri) {
    this.baseUri = baseUri;
    return this;
  }

  public TypesCatalogBuilder addTypesResolver(Consumer<TypesResolverBuilder> typesResolverBuilderConsumer) {
    TypesResolverBuilder typesResolverBuilder = new TypesResolverBuilder(baseUri);
    typesResolverBuilderConsumer.accept(typesResolverBuilder);
    typesResolverBuilders.add(typesResolverBuilder);
    return this;
  }

  public TypesCatalog build() throws Exception {
    return new TypesCatalog(typesResolverBuilders.stream().map(typesResolverBuilder -> {
      try {
        return typesResolverBuilder.build();
      } catch (Exception e) {
        final Optional<Notifier> notifierOptional = typesCatalogBuilderContext.getNotifier();
        if (notifierOptional.isPresent()) {
          notifierOptional.get()
              .reportWarning(
                             NotificationMessages.MSG_FAILED_TO_DECLARE_TYPE_ON_CATALOG(typesResolverBuilder.getName(), e));
        } else {
          Throwables.propagate(e);
        }
        return null;
      }
    }).filter(typeResolver -> typeResolver != null).collect(Collectors.toList()));
  }
}
