package org.mule.datasense.common.loader.xml;

import com.google.common.base.Preconditions;
import org.w3c.dom.Attr;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;

import javax.xml.namespace.QName;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.IntStream;
import java.util.stream.Stream;

public class XmlMatcher {

  private final Element element;

  public static Optional<XmlMatcher> match(Element element, QName qName) {
    Optional<XmlMatcher> result = Optional.empty();
    if (Objects.equals(element.getNamespaceURI(), qName.getNamespaceURI())
        && Objects.equals(element.getLocalName(), qName.getLocalPart())) {
      result = Optional.of(new XmlMatcher(element));
    }
    return result;
  }

  private Stream<Node> nodeStream(NodeList nodeList) {
    return IntStream.range(0, nodeList.getLength()).mapToObj(nodeList::item);
  }

  private XmlMatcher(Element element) {
    Preconditions.checkNotNull(element);
    this.element = element;
  }

  public Optional<XmlMatcher> match(QName qName) {
    if (qName == null) {
      return Optional.empty();
    }
    return matchMany(qName).findFirst();
  }

  public Stream<XmlMatcher> matchAll() {
    return nodeStream(element.getChildNodes())
        .filter(node -> node instanceof Element)
        .map(node -> (Element) node)
        .map(XmlMatcher::new);
  }

  public Optional<XmlMatcher> matchFirst() {
    return matchAll().findFirst();
  }

  public String value() {
    StringBuilder stringBuilder = new StringBuilder();
    nodeStream(element.getChildNodes())
        .filter(node -> node instanceof Text)
        .map(node -> ((Text) node).getTextContent())
        .forEach(stringBuilder::append);
    return stringBuilder.toString();
  }

  public Stream<XmlMatcher> matchMany(QName qName) {
    return nodeStream(element.getElementsByTagNameNS(qName.getNamespaceURI(), qName.getLocalPart()))
        .filter(node -> node instanceof Element)
        .map(node -> (Element) node)
        .filter(element -> qName.equals(new QName(element.getNamespaceURI(), element.getLocalName())))
        .map(XmlMatcher::new);
  }

  private Optional<Attr> matchAttributeNode(String namespaceUri, String localPart) {
    final Attr attributeNodeNS = element.getAttributeNodeNS(namespaceUri, localPart);
    return Optional.ofNullable(attributeNodeNS);
  }

  private Optional<String> matchAttribute(String namespaceUri, String localPart) {
    final Attr attributeNodeNS = matchAttributeNode(namespaceUri, localPart).orElse(null);
    return Optional.ofNullable(attributeNodeNS == null ? null : attributeNodeNS.getValue());
  }

  public Optional<String> matchAttribute(QName qName) {
    return matchAttribute(qName.getNamespaceURI(), qName.getLocalPart());
  }

  public Optional<String> matchAttribute(String localPart) {
    return matchAttribute(null, localPart);
  }

  public Optional<Attr> matchAttributeNode(String localPart) {
    return matchAttributeNode(null, localPart);
  }

  public String requireAttribute(QName qName) {
    return matchAttribute(qName).orElseThrow(IllegalArgumentException::new);
  }

  public String requireAttribute(String localPart) {
    return matchAttribute(localPart).orElseThrow(IllegalArgumentException::new);
  }

  public Element element() {
    return element;
  }

  public static Optional<QName> resolveQName(Attr attr) {
    QName result = null;
    String value = attr.getTextContent();
    final String[] split = value.split(":");
    if (split.length == 2) {
      String prefix = split[0];
      String uri = attr.lookupNamespaceURI(prefix);
      if (uri != null) {
        String localName = split[1];
        result = new QName(uri, localName);
      }
    }
    return Optional.ofNullable(result);
  }
}
