package org.mule.datasense.catalog.loader.xml;

import com.google.common.base.Throwables;
import org.mule.datasense.catalog.builder.TypesCatalogBuilder;
import org.mule.datasense.catalog.loader.BaseTypesCatalogLoader;
import org.mule.datasense.catalog.loader.TypesCatalogLoaderContext;
import org.mule.datasense.common.loader.xml.XmlMatcher;
import org.mule.datasense.common.loader.xml.XmlUtils;
import org.w3c.dom.Element;
import org.xml.sax.InputSource;

import javax.xml.namespace.QName;
import java.io.File;
import java.io.FileInputStream;
import java.io.StringReader;
import java.util.List;

public class TypesCatalogXmlLoader extends BaseTypesCatalogLoader {

  private static final String NS_MULE = "http://www.mulesoft.org/schema/mule/types";
  private static final String NS_TYPES = "http://www.mulesoft.org/schema/mule/types";
  private static final QName ELEM_MULE = new QName(NS_TYPES, "mule");
  private static final QName ELEM_CATALOG = new QName(NS_TYPES, "catalog");
  private static final QName ELEM_TYPE = new QName(NS_TYPES, "type");
  private static final String ELEM_TYPE_ATTR_NAME = "name";
  private static final String ELEM_TYPE_ATTR_FORMAT = "format";
  private static final QName ELEM_SHAPE = new QName(NS_TYPES, "shape");
  private static final String ELEM_SHAPE_ATTR_FORMAT = "format";
  private static final String ELEM_SHAPE_ATTR_LOCATION = "location";
  private static final String ELEM_SHAPE_ATTR_ELEMENT = "element";
  private static final QName ELEM_EXAMPLE = new QName(NS_TYPES, "example");
  private static final String ELEM_EXAMPLE_ATTR_FORMAT = "format";
  private static final String ELEM_EXAMPLE_ATTR_LOCATION = "location";
  private static final String ELEM_EXAMPLE_ATTR_ELEMENT = "element";

  @Override
  protected void loadFiles(List<File> files, TypesCatalogLoaderContext typesCatalogLoaderContext) {
    files.forEach(file -> load(file, typesCatalogLoaderContext));
  }

  protected void load(File file, TypesCatalogLoaderContext typesCatalogLoaderContext) {
    try {
      typesCatalogLoaderContext.getTypesCatalogBuilder().baseUri(file.toURI());
      try (FileInputStream fileInputStream = new FileInputStream(file)) {
        final Element documentElement = XmlUtils.parseRootElement(new InputSource(fileInputStream), true, false);
        load(documentElement, typesCatalogLoaderContext);
      }
    } catch (Exception e) {
      Throwables.propagate(e);
    }
  }

  @Override
  protected void loadData(List<String> data, TypesCatalogLoaderContext typesCatalogLoaderContext) {
    data.forEach(datum -> load(datum, typesCatalogLoaderContext));
  }

  protected void load(String data, TypesCatalogLoaderContext typesCatalogLoaderContext) {
    try {
      load(XmlUtils.parseRootElement(new InputSource(new StringReader(data)), true, false), typesCatalogLoaderContext);
    } catch (Exception e) {
      Throwables.propagate(e);
    }
  }

  private void load(Element documentElement, TypesCatalogLoaderContext typesCatalogLoaderContext) {
    try {
      final TypesCatalogBuilder typesCatalogBuilder = typesCatalogLoaderContext.getTypesCatalogBuilder();
      XmlMatcher.match(documentElement, ELEM_MULE).ifPresent(xmlMatcher -> {
        xmlMatcher.matchMany(ELEM_CATALOG).forEach(catalog -> {
          catalog.matchMany(ELEM_TYPE).forEach(type -> {
            typesCatalogBuilder.addTypesResolver(typesResolverBuilder -> {
              type.matchAttribute(ELEM_TYPE_ATTR_NAME).ifPresent(typesResolverBuilder::name);
              type.matchAttribute(ELEM_TYPE_ATTR_FORMAT).ifPresent(typesResolverBuilder::format);
              type.match(ELEM_SHAPE).ifPresent(shape -> {
                shape.matchAttribute(ELEM_SHAPE_ATTR_LOCATION).ifPresent(typesResolverBuilder::shapeLocation);
                shape.matchAttribute(ELEM_SHAPE_ATTR_FORMAT).ifPresent(typesResolverBuilder::shapeFormat);
                shape.matchAttributeNode(ELEM_SHAPE_ATTR_ELEMENT).ifPresent(elementAttr -> {
                  final QName elementQName =
                      XmlMatcher.resolveQName(elementAttr).orElse(QName.valueOf(elementAttr.getTextContent()));
                  typesResolverBuilder.shapeElement(elementQName.toString());
                });
                final String content = shape.value();
                if (!content.isEmpty()) {
                  typesResolverBuilder.shapeContent(content);
                }
              });
              type.match(ELEM_EXAMPLE).ifPresent(example -> {
                example.matchAttribute(ELEM_EXAMPLE_ATTR_LOCATION)
                    .ifPresent(typesResolverBuilder::exampleLocation);
                example.matchAttribute(ELEM_EXAMPLE_ATTR_FORMAT).ifPresent(typesResolverBuilder::exampleFormat);
                example.matchAttributeNode(ELEM_EXAMPLE_ATTR_ELEMENT).ifPresent(elementAttr -> {
                  final QName elementQName =
                      XmlMatcher.resolveQName(elementAttr).orElse(QName.valueOf(elementAttr.getTextContent()));
                  typesResolverBuilder.exampleElement(elementQName.toString());
                });
                final String content = example.value();
                if (!content.isEmpty()) {
                  typesResolverBuilder.exampleContent(content);
                }
              });
            });
          });
        });

      });
    } catch (Exception e) {
      Throwables.propagate(e);
    }
  }
}
