package org.mule.datasense.catalog.loader.json;

import com.google.common.base.Throwables;
import com.google.gson.JsonElement;
import com.google.gson.JsonParser;
import org.mule.datasense.catalog.builder.TypesCatalogBuilder;
import org.mule.datasense.catalog.loader.BaseTypesCatalogLoader;
import org.mule.datasense.catalog.loader.TypesCatalogLoaderContext;
import org.mule.datasense.common.loader.json.JsonMatcher;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.util.List;

public class TypesCatalogJsonLoader extends BaseTypesCatalogLoader {

  private static final String FIELD_CATALOGS = "catalogs";
  private static final String FIELD_TYPES = "types";
  private static final String FIELD_NAME = "name";
  private static final String FIELD_SHAPE = "shape";
  private static final String FIELD_EXAMPLE = "example";
  private static final String FIELD_LOCATION = "location";
  private static final String FIELD_FORMAT = "format";

  @Override
  protected void loadFiles(List<File> files, TypesCatalogLoaderContext typesCatalogLoaderContext) {
    files.forEach(file -> load(file, typesCatalogLoaderContext));
  }

  protected void load(File file, TypesCatalogLoaderContext typesCatalogLoaderContext) {
    final TypesCatalogBuilder typesCatalogBuilder = typesCatalogLoaderContext.getTypesCatalogBuilder();
    try {
      typesCatalogBuilder.baseUri(file.toURI());
      JsonElement rootElement = new JsonParser().parse(new BufferedReader(new FileReader(file)));
      JsonMatcher.createMatcher(rootElement).ifPresent(jsonMatcher -> {
        jsonMatcher.matchMany(FIELD_CATALOGS).ifPresent(catalogs -> {
          catalogs.forEach(catalog -> {
            catalog.matchMany(FIELD_TYPES).ifPresent(types -> {
              types.forEach(type -> {
                typesCatalogBuilder.addTypesResolver(typesResolverBuilder -> {
                  type.matchString(FIELD_NAME).ifPresent(typesResolverBuilder::name);
                  type.matchString(FIELD_FORMAT).ifPresent(typesResolverBuilder::format);
                  type.match(FIELD_SHAPE).ifPresent(shape -> {
                    shape.matchString(FIELD_LOCATION).ifPresent(typesResolverBuilder::shapeLocation);
                    shape.matchString(FIELD_FORMAT).ifPresent(typesResolverBuilder::shapeFormat);
                  });
                  type.match(FIELD_EXAMPLE).ifPresent(example -> {
                    example.matchString(FIELD_LOCATION).ifPresent(typesResolverBuilder::exampleLocation);
                    example.matchString(FIELD_FORMAT).ifPresent(typesResolverBuilder::exampleFormat);
                  });
                });
              });
            });
          });
        });
      });
    } catch (FileNotFoundException e) {
      Throwables.propagate(e);
    }
  }

  @Override
  protected void loadData(List<String> data, TypesCatalogLoaderContext typesCatalogLoaderContext) {
    throw new UnsupportedOperationException();
  }
}
