package org.mule.datasense.catalog.builder;

import org.apache.commons.io.FileUtils;
import org.mule.datasense.catalog.model.resolver.ExampleTypeResolver;
import org.mule.datasense.catalog.model.resolver.ShapeTypeResolver;
import org.mule.datasense.catalog.model.resolver.SingleTypeResolver;
import org.mule.datasense.catalog.model.resolver.TypeResolver;
import org.mule.datasense.catalog.model.resolver.loaders.TypeLoaderFactory;
import org.mule.datasense.catalog.model.resolver.loaders.TypeLoaderRegistry;
import org.mule.metadata.api.TypeLoader;

import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;

public class TypesResolverBuilder {

  private final URI baseUri;

  private String name;
  private String location;
  private String shapeFormat;
  private String shapeLocation;
  private String shapeContent;
  private String exampleFormat;
  private String exampleLocation;
  private String exampleContent;
  private String format;
  private String shapeElement;
  private String exampleElement;

  public TypesResolverBuilder(URI baseUri) {
    this.baseUri = baseUri;
  }

  public void name(String name) {
    this.name = name;
  }

  public String getName() {
    return name;
  }

  public void format(String format) {
    this.format = format;
  }

  public void shapeFormat(String shapeFormat) {
    this.shapeFormat = shapeFormat;
  }

  public void shapeLocation(String shapeLocation) {
    this.shapeLocation = shapeLocation;
  }

  public void shapeContent(String shapeContent) {
    this.shapeContent = shapeContent;
  }

  public void shapeElement(String shapeElement) {
    this.shapeElement = shapeElement;
  }

  public void exampleFormat(String exampleFormat) {
    this.exampleFormat = exampleFormat;
  }

  public void exampleLocation(String exampleLocation) {
    this.exampleLocation = exampleLocation;
  }

  public void exampleElement(String exampleElement) {
    this.exampleElement = exampleElement;
  }

  public void exampleContent(String exampleContent) {
    this.exampleContent = exampleContent;
  }

  private static URI resolveURI(String location, URI baseUri) throws URISyntaxException {
    if (location == null) {
      return null;
    }
    URI locationURI = new URI(null, null, location, null, null);
    return baseUri != null ? baseUri.resolve(locationURI) : locationURI;
  }

  private String findTypeName(String name, String nativeName) {
    return nativeName != null ? nativeName : name;
  }

  private static TypeLoaderFactory getTypeLoaderFactory(String typeFormat, String shapeFormat,
                                                        TypeLoaderRegistry typeLoaderRegistry) {
    final TypeLoaderFactory typeLoaderFactory = typeLoaderRegistry.getFactory(typeFormat, shapeFormat)
        .orElseThrow(() -> new RuntimeException(String.format("Unsupported typeFormat %s %s Format %s", typeFormat,
                                                              typeLoaderRegistry.getName(),
                                                              shapeFormat)));
    return typeLoaderFactory;
  }

  private static TypeLoader createTypeLoader(String location, String content, TypeLoaderFactory typeLoaderFactory, URI baseUri)
      throws URISyntaxException, MalformedURLException {
    return location != null ? typeLoaderFactory.createTypeLoader(FileUtils.toFile(resolveURI(location, baseUri).toURL()))
        : typeLoaderFactory.createTypeLoader(content);

  }

  public TypeResolver build(TypesCatalogBuilderContext typesCatalogBuilderContext) throws Exception {
    ShapeTypeResolver shapeTypeResolver = null;

    if (shapeFormat != null) {
      final TypeLoaderFactory typeLoaderFactory =
          getTypeLoaderFactory(format, shapeFormat, typesCatalogBuilderContext.getShapeTypeLoaderRegistry());
      final TypeLoader typeLoader = createTypeLoader(shapeLocation, shapeContent, typeLoaderFactory, baseUri);
      shapeTypeResolver = new ShapeTypeResolver(findTypeName(name, shapeElement), shapeFormat, typeLoader);
    }

    ExampleTypeResolver exampleTypeResolver = null;
    if (exampleFormat != null) {
      final TypeLoaderFactory typeLoaderFactory =
          getTypeLoaderFactory(format, exampleFormat, typesCatalogBuilderContext.getExampleTypeLoaderRegistry());
      final TypeLoader typeLoader = createTypeLoader(exampleLocation, exampleContent, typeLoaderFactory, baseUri);
      exampleTypeResolver = new ExampleTypeResolver(exampleElement, exampleFormat, typeLoader);
    }

    return new SingleTypeResolver(name, format, shapeTypeResolver, exampleTypeResolver);
  }
}
