package org.mule.datasense.api.metadataprovider;

import org.mule.runtime.api.exception.MuleRuntimeException;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLConnection;
import java.util.Optional;
import java.util.Properties;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import org.apache.commons.io.IOUtils;

public class ApplicationModelFromUrl extends BaseApplicationModelFactory {

  private static final int CONNECT_TIMEOUT = 5000;
  private static final int READ_TIMEOUT = 5000;
  private static final String MULE_DEPLOY_PROPERTIES = "mule-deploy.properties";
  private static final String CONFIG_RESOURCES = "config.resources";

  private byte[] readToBuffer(URL url) throws IOException {
    InputStream inputStream = null;
    try {
      URLConnection urlConnection = url.openConnection();
      urlConnection.setConnectTimeout(CONNECT_TIMEOUT);
      urlConnection.setReadTimeout(READ_TIMEOUT);
      inputStream = new BufferedInputStream(urlConnection.getInputStream());
      ByteArrayOutputStream bufferOutputStream = new ByteArrayOutputStream();
      IOUtils.copy(inputStream, bufferOutputStream);
      return bufferOutputStream.toByteArray();
    } finally {
      IOUtils.closeQuietly(inputStream);
    }
  }

  private Optional<Stream<String>> findConfigsFromDeployProperties(byte[] buffer) throws IOException {
    Optional<Stream<String>> configsOptional = Optional.empty();
    ZipInputStream zipInputStream = new ZipInputStream(new ByteArrayInputStream(buffer));
    ZipEntry entry;
    while ((entry = zipInputStream.getNextEntry()) != null) {
      if (MULE_DEPLOY_PROPERTIES.equals(entry.getName())) {
        Properties properties = new Properties();
        properties.load(zipInputStream);
        String configResources = properties.getProperty(CONFIG_RESOURCES);
        if (configResources != null) {
          configsOptional = Optional.of(Stream.of(org.apache.commons.lang3.StringUtils.split(configResources, ",")));
        }
        break;
      }
    }
    return configsOptional;
  }

  private Optional<org.mule.runtime.config.spring.api.dsl.model.ApplicationModel> readConfigs(Stream<String> configs,
                                                                                              byte[] buffer)
      throws IOException {
    Optional<org.mule.runtime.config.spring.api.dsl.model.ApplicationModel> result = Optional.empty();
    Set<String> configsSet = configs.collect(Collectors.toSet());
    ZipInputStream zipInputStream = new ZipInputStream(new ByteArrayInputStream(buffer));
    ZipEntry entry;
    while ((entry = zipInputStream.getNextEntry()) != null) {
      // TODO: 11/15/16 first one
      if (configsSet.contains(entry.getName())) {
        result =
            loadConfigLines(zipInputStream)
                .map(configLine -> {
                  try {
                    return loadApplicationModel(configLine);
                  } catch (Exception e) {
                    throw new MuleRuntimeException(e);
                  }
                });
        break;
      }
    }
    return result;
  }

  private Optional<org.mule.runtime.config.spring.api.dsl.model.ApplicationModel> processZip(byte[] buffer) throws IOException {
    return findConfigsFromDeployProperties(buffer).map(configs -> {
      try {
        return readConfigs(configs, buffer);
      } catch (IOException e) {
        throw new MuleRuntimeException(e);
      }
    }).orElse(Optional.empty());
  }

  public Optional<ApplicationModel> create(URL url) throws Exception {
    return processZip(readToBuffer(url)).map(applicationModel -> new DefaultApplicationModel(url.getFile(), applicationModel));
  }
}
