package org.mule.datasense.api.metadataprovider;

import org.mule.datasense.enrichment.model.IdComponentModelSelector;
import org.mule.datasense.impl.DataSenseApplicationModel;
import org.mule.datasense.impl.model.ast.AstNotification;
import org.mule.runtime.api.component.location.Location;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.api.metadata.MetadataKeysContainer;
import org.mule.runtime.api.metadata.descriptor.ComponentMetadataDescriptor;
import org.mule.runtime.api.metadata.resolving.MetadataResult;
import org.mule.runtime.config.spring.api.dsl.model.ComponentModel;
import org.mule.runtime.internal.dsl.DslConstants;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Optional;
import java.util.Set;

public class CachedDataSenseProvider implements DataSenseProvider {

  static final transient Logger logger = LoggerFactory.getLogger(CachedDataSenseProvider.class);
  private final DataSenseMetadataCacheProvider dataSenseMetadataCacheProvider;
  private final DataSenseApplicationModel dataSenseApplicationModel;
  private final ApplicationModel applicationModel;
  private final DataSenseProvider delegate;

  public CachedDataSenseProvider(ApplicationModel applicationModel, DataSenseMetadataCacheProvider dataSenseMetadataCacheProvider,
                                 DataSenseProvider delegate) {
    this.dataSenseApplicationModel = createDataSenseApplicationModel(applicationModel, delegate);
    this.applicationModel = applicationModel;
    this.delegate = delegate;
    this.dataSenseMetadataCacheProvider = dataSenseMetadataCacheProvider;
  }

  @Override
  public Set<ExtensionModel> getExtensions() {
    return delegate.getExtensions();
  }

  private DataSenseApplicationModel createDataSenseApplicationModel(ApplicationModel applicationModel,
                                                                    DataSenseProvider delegate) {
    AstNotification astNotification = new AstNotification();
    final DataSenseApplicationModel dataSenseApplicationModel =
        new DataSenseApplicationModel(null, applicationModel, delegate, astNotification);
    dataSenseApplicationModel.build();
    return dataSenseApplicationModel;
  }

  @Override
  public Optional<DataSenseMetadataProvider> getDataSenseMetadataProvider() {
    return delegate
        .getDataSenseMetadataProvider()
        .map(dataSenseMetadataProvider -> new CachedDataSenseMetadataProvider(dataSenseMetadataProvider,
                                                                              dataSenseMetadataCacheProvider));
  }

  private class CachedDataSenseMetadataProvider implements DataSenseMetadataProvider {

    private final DataSenseMetadataProvider delegate;
    private final DataSenseMetadataCacheProvider dataSenseMetadataCacheProvider;

    public CachedDataSenseMetadataProvider(DataSenseMetadataProvider delegate,
                                           DataSenseMetadataCacheProvider dataSenseMetadataCacheProvider) {
      this.delegate = delegate;
      this.dataSenseMetadataCacheProvider = dataSenseMetadataCacheProvider;
    }

    public DataSenseMetadataProvider getDelegate() {
      return delegate;
    }

    @Override
    public MetadataResult<ComponentMetadataDescriptor<OperationModel>> getOperationMetadata(Location location) {
      try {
        final ComponentModel componentModel = getComponentModel(location);
        return dataSenseMetadataCacheProvider.getOperationMetadata(getComponentId(componentModel), location,
                                                                   resolveEffectiveTimestamp(componentModel), () -> Optional
                                                                       .ofNullable(getDelegate()
                                                                           .getOperationMetadata(location))
                                                                       .orElseThrow(() -> new RuntimeException(String
                                                                           .format("Failed to resolve operation metadata for component path %s.",
                                                                                   location))));
      } catch (Exception e) {
        logger.error(String.format("Failed to resolve operation metadata for component path %s.", location), e);
        return null;
      }
    }

    @Override
    public MetadataResult<ComponentMetadataDescriptor<SourceModel>> getSourceMetadata(Location location) {
      try {
        final ComponentModel componentModel = getComponentModel(location);
        return dataSenseMetadataCacheProvider
            .getSourceMetadata(getComponentId(componentModel), location, resolveEffectiveTimestamp(componentModel),
                               () -> Optional.ofNullable(getDelegate().getSourceMetadata(location))
                                   .orElseThrow(() -> new RuntimeException(String
                                       .format("Failed to resolve source metadata for component path %s.", location))));
      } catch (Exception e) {
        logger.error(String.format("Failed to resolve source metadata for component path %s.", location), e);
        return null;
      }
    }

    @Override
    public MetadataResult<MetadataKeysContainer> getMetadataKeys(Location location) {
      try {
        final ComponentModel componentModel = getComponentModel(location);
        return dataSenseMetadataCacheProvider
            .getMetadataKeys(getComponentId(componentModel), location, resolveEffectiveTimestamp(componentModel),
                             () -> Optional.ofNullable(getDelegate().getMetadataKeys(location))
                                 .orElseThrow(() -> new RuntimeException(String
                                     .format("Failed to resolve metadata keys for component path %s.", location))));
      } catch (Exception e) {
        logger.error(String.format("Failed to resolve metadata keys for component path %s.", location), e);
        return null;
      }
    }

    private Optional<Long> resolveTimestamp(ComponentModel componentModel) {
      if (componentModel == null) {
        return Optional.empty();
      }
      try {
        return Optional.of(Long.parseLong(componentModel.getParameters().get("doc:timestamp")));
      } catch (NumberFormatException e) {
        return Optional.empty();
      }
    }

    private Optional<String> getConfigurationRef(ComponentModel componentModel) {
      return Optional.ofNullable(componentModel.getParameters().get(DslConstants.CONFIG_ATTRIBUTE_NAME));
    }

    private ComponentModel getComponentModel(Location location) {
      return dataSenseApplicationModel.find(location)
          .orElseThrow(() -> new RuntimeException(String.format("Component path %s not found on app.", location)));
    }

    private String getComponentId(ComponentModel componentModel) {
      return IdComponentModelSelector.getComponentId(componentModel);
    }

    private Long resolveEffectiveTimestamp(ComponentModel componentModel) {
      final Long componentTimestamp = resolveTimestamp(componentModel).orElse(null);
      final Optional<ComponentModel> relatedConfigurationOptional =
          getConfigurationRef(componentModel).flatMap(applicationModel::findNamedComponent);
      if (relatedConfigurationOptional.isPresent()) {
        final Long configurationTimestamp = resolveTimestamp(relatedConfigurationOptional.get()).orElse(null);
        return componentTimestamp != null && configurationTimestamp != null ? Long.max(componentTimestamp, configurationTimestamp)
            : null;
      } else {
        return componentTimestamp;
      }
    }
  }
}
