/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.datasense.impl;

import org.mule.datasense.api.metadataprovider.ApplicationModel;
import org.mule.datasense.api.metadataprovider.DataSenseProvider;
import org.mule.datasense.catalog.builder.TypesCatalogBuilderContext;
import org.mule.datasense.catalog.loader.xml.TypesCatalogXmlLoader;
import org.mule.datasense.catalog.model.TypesCatalog;
import org.mule.datasense.common.loader.xml.XmlUtils;
import org.mule.datasense.common.util.Notifier;
import org.mule.datasense.declarations.loader.TypeDeclarationLoaderContext;
import org.mule.datasense.declarations.loader.xml.ExtensionOperationTypeDeclarationXmlLoader;
import org.mule.datasense.declarations.loader.xml.MessageProcessorTypeDeclarationXmlLoader;
import org.mule.datasense.declarations.loader.xml.TypeDeclarationXmlLoader;
import org.mule.datasense.declarations.model.ExtensionOperationTypeDeclaration;
import org.mule.datasense.declarations.model.MessageProcessorTypeDeclaration;
import org.mule.datasense.enrichment.loader.ComponentModelEnrichmentLoaderContext;
import org.mule.datasense.enrichment.loader.xml.ComponentModelEnrichmentXmlLoader;
import org.mule.datasense.enrichment.model.ComponentModelEnrichments;
import org.mule.datasense.impl.model.ast.AstNotification;
import org.mule.datasense.impl.model.ast.MessageProcessorNode;
import org.mule.datasense.impl.model.reporting.NotificationMessages;
import org.mule.datasense.impl.util.DefaultDslResolvingContext;
import org.mule.module.apikit.metadata.Metadata;
import org.mule.module.apikit.metadata.interfaces.ResourceLoader;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.i18n.I18nMessage;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.config.ConfigurationModel;
import org.mule.runtime.api.meta.model.operation.HasOperationModels;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.source.HasSourceModels;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.config.spring.api.dsl.model.ComponentModel;
import org.mule.runtime.config.spring.api.dsl.model.DslElementModelFactory;
import org.mule.runtime.internal.dsl.DslConstants;

import com.google.common.base.Throwables;

import java.io.File;
import java.io.StringReader;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.w3c.dom.Element;
import org.xml.sax.InputSource;

/**
 *
 */
public class DataSenseProviderResolver {

  private static final List<ComponentIdentifier> componentIdentifierBlackList = Collections.singletonList(ComponentIdentifier
      .builder().namespace("http://www.mulesoft.org/schema/mule/http").name("listener").build());

  public static boolean isDynamicDataSenseSupportedFor(MessageProcessorNode messageProcessorNode) {
    return !componentIdentifierBlackList.contains(messageProcessorNode.getComponentIdentifier());
  }

  private final DataSenseProvider dataSenseProvider;
  private final ApplicationModelResolver applicationModelResolver;
  private final AstNotification astNotification;
  private final Notifier notifier;
  private final DslElementModelFactory dslElementModelFactory;
  private Map<String, ExtensionModel> extensionModelsByUri;
  private final TypesCatalog typesCatalog;
  private final MessageProcessorTypeDeclarationXmlLoader messageProcessorTypeDeclarationXmlLoader;
  private final ExtensionOperationTypeDeclarationXmlLoader extensionOperationTypeDeclarationXmlLoader;

  private final ComponentModelEnrichments componentModelEnrichments;
  private Metadata apiKitMetadata = null;

  public DataSenseProviderResolver(DataSenseProvider dataSenseProvider, ApplicationModelResolver applicationModelResolver,
                                   AstNotification astNotification) {
    this.dataSenseProvider = dataSenseProvider;
    this.applicationModelResolver = applicationModelResolver;
    this.astNotification = astNotification;
    notifier = new Notifier() {

      @Override
      public void reportWarning(I18nMessage i18nMessage) {
        astNotification.reportWarning(i18nMessage);
      }

      @Override
      public void reportError(I18nMessage i18nMessage) {
        astNotification.reportError(i18nMessage);
      }

      @Override
      public void reportFatalError(I18nMessage i18nMessage) {
        astNotification.reportFatalError(i18nMessage);
      }
    };
    extensionModelsByUri = new HashMap<>();

    Set<ExtensionModel> extensions = dataSenseProvider.getExtensions()
        .stream()
        .filter(Objects::nonNull)
        .filter(extensionModel -> extensionModel.getXmlDslModel() != null).collect(Collectors.toSet());

    extensions
        .forEach(extensionModel -> {
          extensionModelsByUri.put(extensionModel.getXmlDslModel().getNamespace(), extensionModel);
        });

    dslElementModelFactory = DslElementModelFactory.getDefault(new DefaultDslResolvingContext(extensions));

    final ApplicationModel applicationModel = applicationModelResolver.getApplicationModel();
    final Optional<String> typesDataOptional = applicationModel.findTypesData();
    typesCatalog = typesDataOptional.map(typesData -> {
      try {
        TypesCatalogBuilderContext typesCatalogBuilderContext = new TypesCatalogBuilderContext(notifier);
        return new TypesCatalogXmlLoader().load(typesData, typesCatalogBuilderContext,
                                                applicationModel.findResource("").orElse(null));
      } catch (Exception e) {
        Throwables.propagate(e);
        return null;
      }
    }).orElse(null);

    componentModelEnrichments = typesDataOptional.flatMap(typesData -> {
      final Element element;
      try {
        element = XmlUtils.parseRootElement(new InputSource(new StringReader(typesData)), true, false);
        ComponentModelEnrichmentXmlLoader componentModelEnrichmentXmlLoader = new ComponentModelEnrichmentXmlLoader();
        return componentModelEnrichmentXmlLoader.load(element, new ComponentModelEnrichmentLoaderContext());
      } catch (Exception e) {
        return null;
      }
    }).orElse(null);
    messageProcessorTypeDeclarationXmlLoader = new MessageProcessorTypeDeclarationXmlLoader();
    extensionOperationTypeDeclarationXmlLoader = new ExtensionOperationTypeDeclarationXmlLoader();

    try {
      buildApiKitMetadata(astNotification, applicationModel);
    } catch (Exception e) {
      astNotification.reportError(null, NotificationMessages.MSG_RAML_RESOLUTION_INITIALIZATION(e.getMessage()));
    }
  }

  private void buildApiKitMetadata(final AstNotification astNotification, final ApplicationModel applicationModel) {
    final Metadata.Builder apiKitMetadataBuilder = new Metadata.Builder();
    apiKitMetadataBuilder.withApplicationModel(applicationModel.getMuleApplicationModel());
    apiKitMetadataBuilder.withResourceLoader(new ResourceLoader() {

      @Override
      public File getRamlResource(String resource) {
        try {
          resource = new URI("api/").resolve(resource).getPath();
        } catch (URISyntaxException e) {
        }
        return applicationModel.findResource(resource).map(uri -> {
          try {
            return new File(uri.toURL().getFile());
          } catch (MalformedURLException e) {
            return null;
          }
        }).orElse(null);
      }
    });
    apiKitMetadataBuilder.withNotifier(new org.mule.module.apikit.metadata.interfaces.Notifier() {

      @Override
      public void error(String s) {
        astNotification.reportError(NotificationMessages.MSG_RAML_RESOLUTION_NOTIFICATION("error", s));
      }

      @Override
      public void warn(String s) {
        astNotification.reportWarning(NotificationMessages.MSG_RAML_RESOLUTION_NOTIFICATION("warning", s));

      }

      @Override
      public void info(String s) {
        astNotification.reportInfo(NotificationMessages.MSG_RAML_RESOLUTION_NOTIFICATION("info", s));

      }

      @Override
      public void debug(String s) {
        astNotification.reportDebug(NotificationMessages.MSG_RAML_RESOLUTION_NOTIFICATION("trace", s));
      }
    });
    apiKitMetadata = apiKitMetadataBuilder.build();
  }

  public DataSenseProvider getDataSenseProvider() {
    return dataSenseProvider;
  }

  public ApplicationModelResolver getApplicationModelResolver() {
    return applicationModelResolver;
  }

  private Optional<String> getConfigurationRef(ComponentModel componentModel) {
    return Optional.ofNullable(componentModel.getParameters().get(DslConstants.CONFIG_ATTRIBUTE_NAME));
  }

  public Optional<TypesCatalog> getTypesCatalog() {
    return Optional.ofNullable(typesCatalog);
  }

  public Optional<ComponentModelEnrichments> getComponentModelEnrichments() {
    return Optional.ofNullable(componentModelEnrichments);
  }

  /**
   * @param componentModel
   * @return
   */
  public Optional<ExtensionModel> findExtensionModel(ComponentModel componentModel) {
    String namespaceUri = (String) componentModel.getCustomAttributes().get("NAMESPACE_URI");
    return Optional.ofNullable(extensionModelsByUri.get(namespaceUri));
  }

  private Optional<ConfigurationModel> getConfigurationModel(String configuration, ExtensionModel extensionModel) {
    Optional<ConfigurationModel> configurationModel = Optional.empty();
    Optional<ComponentModel> namedComponent =
        getApplicationModelResolver().getApplicationModel().findNamedComponent(configuration);
    if (namedComponent.isPresent()) {
      configurationModel =
          extensionModel.getConfigurationModel(namedComponent.get().getIdentifier().getName());
    }
    return configurationModel;
  }

  private Optional<? extends HasSourceModels> getHasSourceModels(ComponentModel componentModel, ExtensionModel extensionModel) {
    Optional<String> configurationRef = getConfigurationRef(componentModel);
    return configurationRef.isPresent() ? getConfigurationModel(configurationRef.get(), extensionModel)
        : Optional.of(extensionModel);
  }

  /**
   * @param componentModel
   * @return
   */
  public Optional<SourceModel> resolveSourceModel(ComponentModel componentModel) {
    Optional<SourceModel> result = Optional.empty();
    Optional<ExtensionModel> optionalExtensionModel = findExtensionModel(componentModel);
    if (optionalExtensionModel.isPresent()) {
      ExtensionModel extensionModel = optionalExtensionModel.get();
      Optional<? extends HasSourceModels> hasSourceModels = getHasSourceModels(componentModel, extensionModel);
      if (hasSourceModels.isPresent()) {
        result =
            hasSourceModels.get().getSourceModel(componentModel.getIdentifier().getName());
      }
    }
    return result;
  }

  private Optional<? extends HasOperationModels> getHasOperationModels(ComponentModel componentModel,
                                                                       ExtensionModel extensionModel) {
    Optional<String> configurationRef = getConfigurationRef(componentModel);
    return configurationRef.isPresent() ? getConfigurationModel(configurationRef.get(), extensionModel)
        : Optional.of(extensionModel);
  }

  /**
   * @param componentModel
   * @return
   */
  public Optional<OperationModel> resolveOperationModel(ComponentModel componentModel) {
    Optional<OperationModel> result = Optional.empty();
    Optional<ExtensionModel> optionalExtensionModel = findExtensionModel(componentModel);
    if (optionalExtensionModel.isPresent()) {
      ExtensionModel extensionModel = optionalExtensionModel.get();
      Optional<? extends HasOperationModels> hasOperationModels = getHasOperationModels(componentModel, extensionModel);
      if (hasOperationModels.isPresent()) {
        result =
            hasOperationModels.get().getOperationModel(componentModel.getIdentifier().getName());
      }
    }
    return result;
  }

  public DslElementModelFactory getDslElementModelFactory() {
    return dslElementModelFactory;
  }

  private <T> Optional<T> findTypeDeclaration(ComponentModel componentModel,
                                              TypeDeclarationXmlLoader<T, Element> typeDeclarationLoader) {
    return getComponentModelEnrichments()
        .flatMap(componentModelEnrichments1 -> componentModelEnrichments1.enrich(componentModel).map(o -> (Element) o)
            .map(element -> typeDeclarationLoader.load(element,
                                                       new TypeDeclarationLoaderContext(componentModel, typesCatalog, notifier)))
            .filter(Optional::isPresent)
            .map(Optional::get)
            .findFirst());
  }

  public Optional<MessageProcessorTypeDeclaration> findMessageProcessorTypeDeclaration(ComponentModel componentModel) {
    return findTypeDeclaration(componentModel, messageProcessorTypeDeclarationXmlLoader);
  }

  public Optional<ExtensionOperationTypeDeclaration> findExtensionOperationTypeDeclaration(ComponentModel componentModel) {
    return findTypeDeclaration(componentModel, extensionOperationTypeDeclarationXmlLoader);
  }

  public AstNotification getAstNotification() {
    return astNotification;
  }

  public Optional<Metadata> getApiKitMetadata() {
    return Optional.ofNullable(apiKitMetadata);
  }
}
