package org.mule.datasense.impl.model.operation;

import org.mule.datasense.impl.model.types.TypesHelper;
import org.mule.datasense.impl.util.LogSupport;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.message.MessageMetadataType;
import org.mule.metadata.message.MessageMetadataTypeBuilder;

import java.util.HashMap;
import java.util.Map;
import java.util.function.Consumer;

import static org.mule.datasense.impl.util.extension.ExtensionUtils.isOutputParameter;

public class OperationCallBuilder implements LogSupport {



  public static class InputMappingBuilder {

    private InputParameterBuilder inputParameterBuilder;
    private InputArgumentBuilder inputArgumentBuilder;

    public InputMappingBuilder() {
      inputParameterBuilder = new InputParameterBuilder();
      inputArgumentBuilder = new InputArgumentBuilder();
    }

    public InputMappingBuilder parameter(Consumer<InputParameterBuilder> consumer) {
      consumer.accept(inputParameterBuilder);
      return this;
    }

    public InputMappingBuilder argument(Consumer<InputArgumentBuilder> consumer) {
      consumer.accept(inputArgumentBuilder);
      return this;
    }

    public boolean isOutput() {
      return isOutputParameter(inputParameterBuilder.name);
    }

    public InputMapping build() {
      return new InputMapping(inputParameterBuilder.build(), inputArgumentBuilder.build());
    }
  }

  public static class InputParameterBuilder {

    private String name;
    private MetadataType type;

    public InputParameterBuilder name(String name) {
      this.name = name;
      return this;
    }

    public InputParameterBuilder type(MetadataType type) {
      this.type = type;
      return this;
    }

    public InputParameter build() {
      return new InputParameter(name, type);
    }
  }

  public static class InputArgumentBuilder {

    private String expression;

    public InputArgumentBuilder expression(String expression) {
      this.expression = expression;
      return this;
    }

    public InputArgument build() {
      return new InputArgument(expression);
    }
  }

  private String name;
  private Map<String, InputMappingBuilder> inputMappingsMap;
  private MetadataType returnType;
  private String target;
  private String targetValueExpression;

  public OperationCallBuilder() {
    this.name = "";
    inputMappingsMap = new HashMap<>();
    returnType = null;
  }

  public OperationCallBuilder name(String name) {
    this.name = name;
    return this;
  }

  public OperationCallBuilder target(String target) {
    this.target = target;
    return this;
  }

  public OperationCallBuilder targetValue(String targetValueExpression) {
    this.targetValueExpression = targetValueExpression;
    return this;
  }

  public OperationCallBuilder input(String name, Consumer<InputMappingBuilder> consumer) {
    InputMappingBuilder inputMappingBuilder = inputMappingsMap.get(name);
    if (inputMappingBuilder == null) {
      inputMappingBuilder = new InputMappingBuilder();
    }
    return input(inputMappingBuilder, consumer);
  }

  public OperationCallBuilder input(Consumer<InputMappingBuilder> consumer) {
    return input((String) null, consumer);
  }

  public OperationCallBuilder input(InputMappingBuilder inputMappingBuilder, Consumer<InputMappingBuilder> consumer) {
    consumer.accept(inputMappingBuilder);

    if (inputMappingBuilder.inputParameterBuilder != null) {
      String name = inputMappingBuilder.inputParameterBuilder.name;
      inputMappingsMap.put(name, inputMappingBuilder);
    }
    return this;
  }


  public OperationCallBuilder returnType(MetadataType returnType) {
    MetadataType effectiveMetadataType = returnType;

    if (this.returnType != null) {
      logger().debug("retyping {} with: {}", name, TypesHelper.toString(returnType));
      if (this.returnType instanceof MessageMetadataType && returnType instanceof MessageMetadataType) {
        MessageMetadataType previousReturnType = (MessageMetadataType) this.returnType;
        MessageMetadataType newReturnType = (MessageMetadataType) returnType;

        final MessageMetadataTypeBuilder messageMetadataTypeBuilder = TypesHelper.getMessageMetadataTypeBuilder();

        previousReturnType.getPayloadType().ifPresent(messageMetadataTypeBuilder::payload);
        newReturnType.getPayloadType().ifPresent(messageMetadataTypeBuilder::payload);

        previousReturnType.getAttributesType().ifPresent(messageMetadataTypeBuilder::attributes);
        newReturnType.getAttributesType().ifPresent(messageMetadataTypeBuilder::attributes);

        effectiveMetadataType = messageMetadataTypeBuilder.build();
      }
    }

    this.returnType = effectiveMetadataType;
    return this;
  }

  public OperationCall build() {
    return new OperationCall(name, inputMappingsMap.values().stream()
        .filter(inputMappingBuilder -> !inputMappingBuilder.isOutput()).map(InputMappingBuilder::build), returnType, target,
                             targetValueExpression);
  }

}
