package org.mule.datasense.impl.phases.builder;

import org.mule.datasense.api.ApplicationResolutionScope;
import org.mule.datasense.api.ComponentResolutionScope;
import org.mule.datasense.api.ConfigResolutionScope;
import org.mule.datasense.api.DataSenseResolutionScope;
import org.mule.datasense.impl.phases.typing.resolver.TypeResolver;
import org.mule.datasense.impl.phases.typing.resolver.TypeResolverRegistry;
import org.mule.datasense.impl.util.ComponentIdentifierUtils;
import org.mule.datasense.impl.util.TreeLogSupport;
import org.mule.runtime.config.spring.api.dsl.model.ComponentModel;

import java.util.Optional;

public class MuleAstParserContext {

  private final TreeLogSupport treeLogSupport;
  private final ComponentModelClassifier componentModelClassifier;
  private final DataSenseResolutionScopeStrategy dataSenseResolutionScopeStrategy;
  private final TypeResolverRegistry typeResolverRegistry;

  public MuleAstParserContext(ComponentModelClassifier componentModelClassifier) {
    this(componentModelClassifier, null, null);

  }

  public MuleAstParserContext(ComponentModelClassifier componentModelClassifier,
                              DataSenseResolutionScope dataSenseResolutionScope, TypeResolverRegistry typeResolverRegistry) {
    this.componentModelClassifier = componentModelClassifier;
    this.dataSenseResolutionScopeStrategy = createDataSenseResolutionScopeStrategy(dataSenseResolutionScope);
    this.typeResolverRegistry = typeResolverRegistry;
    treeLogSupport = new TreeLogSupport();
  }

  public Optional<DataSenseResolutionScopeStrategy> getDataSenseResolutionScopeStrategy() {
    return Optional.ofNullable(dataSenseResolutionScopeStrategy);
  }

  TreeLogSupport astlogger() {
    return treeLogSupport;
  }

  void enter(ComponentModel componentModel) {
    astlogger().enter(componentModel.getIdentifier().toString());
  }

  void exit(ComponentModel componentModel) {
    astlogger().exit(componentModel.getIdentifier().toString());
  }

  public Optional<ComponentModelType> getComponentModelType(ComponentModel componentModel) {
    return componentModelClassifier
        .getComponentModelType(componentModel);
  }

  public Optional<MuleAstParseProvider> getParseProvider(
                                                         ComponentModel componentModel) {
    return typeResolverRegistry.get(ComponentIdentifierUtils
        .createFromComponentModel(componentModel)).flatMap(TypeResolver::getParseProvider);
  }

  private static DataSenseResolutionScopeStrategy createDataSenseResolutionScopeStrategy(DataSenseResolutionScope dataSenseResolutionScope) {
    if (dataSenseResolutionScope == null) {
      return null;
    }

    if (dataSenseResolutionScope instanceof ApplicationResolutionScope) {
      return new ApplicationResolutionScopeStrategy((ApplicationResolutionScope) dataSenseResolutionScope);
    } else if (dataSenseResolutionScope instanceof ConfigResolutionScope) {
      return new ConfigResolutionScopeStrategy((ConfigResolutionScope) dataSenseResolutionScope);
    } else if (dataSenseResolutionScope instanceof ComponentResolutionScope) {
      return new ComponentResolutionScopeStrategy((ComponentResolutionScope) dataSenseResolutionScope);
    } else {
      throw new IllegalArgumentException(String.format("Unknown resolution scope class %s.",
                                                       dataSenseResolutionScope.getClass()));
    }
  }
}
