package org.mule.datasense.impl.model.ast;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.component.location.Location;

import com.google.common.base.Preconditions;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Stream;

;import static java.util.Optional.empty;

public class MuleApplicationNode extends BaseAstNode {

  private final List<MuleFlowNode> muleFlowNodes;
  private final Map<String, MuleFlowNode> muleFlowNodeByNameMap;
  private Map<ComponentLocation, MessageProcessorNode> messageProcessorNodeMap;

  public MuleApplicationNode(ComponentIdentifier componentIdentifier, Stream<MuleFlowNode> muleFlowNodes) {
    super(componentIdentifier);
    this.muleFlowNodes = new ArrayList<>();
    this.muleFlowNodeByNameMap = new HashMap<>();
    messageProcessorNodeMap = new HashMap<>();
    muleFlowNodes.forEach(this::add);
  }

  private void add(MuleFlowNode muleFlowNode) {
    muleFlowNodes.add(muleFlowNode);
    muleFlowNodeByNameMap.put(muleFlowNode.getName(), muleFlowNode);
  }

  public Stream<MuleFlowNode> getMuleFlowNodes() {
    return muleFlowNodes.stream();
  }

  @Override
  public <T> Object accept(AstNodeVisitor<T> astNodeVisitor, T context) {
    return astNodeVisitor.visit(this, context);
  }

  public void register(ComponentLocation componentLocation, MessageProcessorNode messageProcessorNode) {
    Preconditions.checkState(!messageProcessorNodeMap.containsKey(componentLocation),
                             String.format("Node already registered for componentLocation %s", componentLocation));
    messageProcessorNodeMap.put(componentLocation, messageProcessorNode);
  }

  public Optional<MessageProcessorNode> findMessageProcessorNode(String location) {
    return messageProcessorNodeMap.entrySet().stream()
        .filter(messageProcessorNode -> messageProcessorNode.getKey().getLocation().equals(location)).map(
                                                                                                          Map.Entry::getValue)
        .findFirst();
  }

  public Optional<MessageProcessorNode> findMessageProcessorNode(Location location) {
    return findMessageProcessorNode(location.toString());
  }

  public Optional<MuleFlowNode> findMuleFlowNode(String flow) {
    if (flow == null) {
      return empty();
    }

    return getMuleFlowNodes().filter(muleFlowNode -> flow.equals(muleFlowNode.getName())).findFirst();
  }

  public Stream<MessageProcessorNode> findMessageProcessorNodes(boolean includeRoot) {
    return messageProcessorNodeMap.values().stream()
        .filter(messageProcessorNode -> includeRoot || !messageProcessorNode.isRootMessageProcessorNode());
  }

  public Stream<MessageProcessorNode> findMessageProcessorNodes() {
    return findMessageProcessorNodes(false);
  }

}
