package org.mule.datasense.impl.phases.typing.resolver;

import org.mule.datasense.impl.model.annotations.DefinesTypeAnnotation;
import org.mule.datasense.impl.model.annotations.OperationCallAnnotation;
import org.mule.datasense.impl.model.annotations.UsesTypeAnnotation;
import org.mule.datasense.impl.model.annotations.VoidOperationAnnotation;
import org.mule.datasense.impl.model.ast.AstNodeLocation;
import org.mule.datasense.impl.model.ast.AstNotification;
import org.mule.datasense.impl.model.ast.MessageProcessorNode;
import org.mule.datasense.impl.model.operation.InputArgument;
import org.mule.datasense.impl.model.operation.InputParameter;
import org.mule.datasense.impl.model.operation.OperationCall;
import org.mule.datasense.impl.model.reporting.NotificationMessages;
import org.mule.datasense.impl.model.types.EventType;
import org.mule.datasense.impl.model.types.TypeUtils;
import org.mule.datasense.impl.model.types.TypesHelper;
import org.mule.datasense.impl.model.types.VarDecl;
import org.mule.datasense.impl.phases.typing.TypingMuleAstVisitor;
import org.mule.datasense.impl.phases.typing.TypingMuleAstVisitorContext;
import org.mule.datasense.impl.util.ExpressionLanguageUtils;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.message.api.MessageMetadataType;
import org.mule.metadata.message.api.MuleEventMetadataType;
import org.mule.metadata.message.api.MuleEventMetadataTypeBuilder;
import org.mule.metadata.message.api.el.ExpressionLanguageMetadataTypeResolver;

import java.util.Optional;
import java.util.stream.Stream;

public class OperationCallTypeResolver extends SingleNodeTypeResolver {

  protected ExpressionLanguageMetadataTypeResolver.MessageCallback createMessageCallback(AstNotification astNotification,
                                                                                         AstNodeLocation astNodeLocation) {
    return new ExpressionLanguageMetadataTypeResolver.MessageCallback() {

      @Override
      public void warning(String message, ExpressionLanguageMetadataTypeResolver.MessageLocation messageLocation) {
        astNotification.reportWarning(astNodeLocation, NotificationMessages.MSG_SCRIPTING_LANGUAGE_WARNING("", message));
      }

      @Override
      public void error(String message, ExpressionLanguageMetadataTypeResolver.MessageLocation messageLocation) {
        astNotification.reportError(astNodeLocation, NotificationMessages.MSG_SCRIPTING_LANGUAGE_ERROR("", message));
      }
    };
  }



  @Override
  protected EventType resolve(MessageProcessorNode messageProcessorNode, EventType inputEventType,
                              TypingMuleAstVisitor typingMuleAstVisitor,
                              TypingMuleAstVisitorContext visitorContext) {
    Optional<OperationCallAnnotation> annotation = messageProcessorNode.getAnnotation(OperationCallAnnotation.class);
    if (annotation.isPresent()) {
      OperationCallAnnotation operationCallAnnotation = annotation.get();
      OperationCall operationCall = operationCallAnnotation.getOperationCall();

      MuleEventMetadataTypeBuilder inferredMuleEventTypeBuilder = TypesHelper.getMuleEventMetadataTypeBuilder();

      final ExpressionLanguageMetadataTypeResolver expressionLanguageMetadataTypeResolver =
          visitorContext.getExpressionLanguageMetadataTypeResolver();
      final ExpressionLanguageMetadataTypeResolver.MessageCallback messageCallback =
          createMessageCallback(visitorContext.getAstNotification(),
                                messageProcessorNode.getAstNodeLocation());

      operationCall.getInputMappings().forEach(inputMapping -> {
        InputArgument inputArgument = inputMapping.getInputArgument();
        InputParameter inputParameter = inputMapping.getInputParameter();
        if (inputArgument.getExpression() != null) {
          ExpressionLanguageUtils
              .resolveInputEventType(inputArgument.getExpression(), expressionLanguageMetadataTypeResolver,
                                     inputParameter.getMetadataType(),
                                     inferredMuleEventTypeBuilder,
                                     messageCallback);

        }
      });

      MuleEventMetadataType inferredInputMuleEventType = inferredMuleEventTypeBuilder.build();
      messageProcessorNode.annotate(new UsesTypeAnnotation(TypeUtils.asEventType(inferredInputMuleEventType)));

      EventType outputEventType = inputEventType;
      if (!messageProcessorNode.isAnnotatedWith(VoidOperationAnnotation.class)) {
        MetadataType returnType = operationCall.getReturnType();
        if (returnType instanceof MessageMetadataType) {
          final MessageMetadataType returnMessageMetadataType = (MessageMetadataType) returnType;

          final String target = operationCall.getTarget().orElse(null);
          if (target != null) {
            MetadataType targetEventType =
                ExpressionLanguageUtils.extractExpression(operationCall.getTargetValueExpression()).map(targetValueExpression -> {
                  return ExpressionLanguageUtils.resolveExpressionType(targetValueExpression,
                                                                       TypesHelper.getMuleEventMetadataTypeBuilder()
                                                                           .message(returnMessageMetadataType).build(),
                                                                       visitorContext.getTypeBindings(),
                                                                       expressionLanguageMetadataTypeResolver, messageCallback);
                }).orElse(null);
            outputEventType = new EventType(Stream
                .of(new VarDecl(target, targetEventType != null ? targetEventType : returnMessageMetadataType)));
          } else {
            outputEventType = new EventType(TypeUtils.asVarDecls(returnMessageMetadataType, true));
          }

          messageProcessorNode.annotate(new DefinesTypeAnnotation(outputEventType));
        }
      }
      return outputEventType;
    } else {
      // TODO: 11/12/16 handle this
      throw new IllegalStateException();
    }
  }

  @Override
  protected boolean isPropagates(MessageProcessorNode messageProcessorNode) {
    return true;
  }
}
