package org.mule.datasense.impl.phases.annotators;

import org.mule.datasense.impl.model.annotations.FunctionBindingsAnnotation;
import org.mule.datasense.impl.model.ast.MuleApplicationNode;
import org.mule.datasense.impl.model.types.TypesHelper;
import org.mule.datasense.impl.phases.typing.AnnotatingMuleAstVisitorContext;
import org.mule.datasense.impl.phases.typing.resolver.GlobalBindingMetadataTypes;
import org.mule.metadata.api.builder.FunctionTypeBuilder;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.message.api.el.ModuleDefinition;
import org.mule.metadata.message.api.el.TypeBindings;
import org.mule.runtime.api.el.ModuleNamespace;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.function.FunctionModel;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

public class FunctionBindingsAnnotator implements Annotator {

  final String JAVACLASS_TYPEDVALUE = "org.mule.runtime.api.metadata.TypedValue";

  @Override
  public void annotate(MuleApplicationNode muleApplicationNode,
                       AnnotatingMuleAstVisitorContext annotatingMuleAstVisitorContext) {
    final TypeBindings.Builder functionBindingsBuilder = TypeBindings.builder();

    registerMuleFunctions(functionBindingsBuilder, muleApplicationNode, annotatingMuleAstVisitorContext);
    registerExtensionFunctions(functionBindingsBuilder,
                               annotatingMuleAstVisitorContext.getDataSenseProviderResolver().getDataSenseProvider()
                                   .getExtensions());

    muleApplicationNode.annotate(new FunctionBindingsAnnotation(functionBindingsBuilder.build()));
  }

  private void registerMuleFunctions(TypeBindings.Builder functionBindings, MuleApplicationNode muleApplicationNode,
                                     AnnotatingMuleAstVisitorContext annotatingMuleAstVisitorContext) {
    registerMulePropertyFunction(functionBindings);
    registerMuleLookupFunction(functionBindings);
  }

  private void registerMuleLookupFunction(TypeBindings.Builder functionBindings) {
    final FunctionTypeBuilder lookupFunctionTypeBuilder = TypesHelper.getTypeBuilder().functionType();
    lookupFunctionTypeBuilder.addParameterOf("flowName").stringType();
    lookupFunctionTypeBuilder.addParameterOf("payload").anyType();
    lookupFunctionTypeBuilder
        .returnType(GlobalBindingMetadataTypes.getTypeFromJavaClass(JAVACLASS_TYPEDVALUE));
    functionBindings.addBinding(createFunctionIdentifier("lookup"), lookupFunctionTypeBuilder.build());
  }

  private void registerMulePropertyFunction(TypeBindings.Builder functionBindings) {
    final FunctionTypeBuilder propertyFunctionTypeBuilder = TypesHelper.getTypeBuilder().functionType();
    propertyFunctionTypeBuilder.addParameterOf("name").stringType();
    propertyFunctionTypeBuilder.returnType().stringType();
    functionBindings.addBinding(createFunctionIdentifier("p"), propertyFunctionTypeBuilder.build());
  }

  private void registerExtensionFunctions(TypeBindings.Builder functionBindings, Set<ExtensionModel> extensionModels) {
    extensionModels.forEach(extensionModel -> {
      Map<String, MetadataType> functionsMap = new HashMap<>();
      extensionModel.getFunctionModels().forEach(functionModel -> {
        final FunctionTypeBuilder functionTypeBuilder = TypesHelper.getTypeBuilder().functionType();
        functionModel.getAllParameterModels().forEach(parameterModel -> {
          functionTypeBuilder.addParameterOf(parameterModel.getName(), parameterModel.getType());
        });
        functionTypeBuilder.returnType(functionModel.getOutput().getType());
        functionsMap.put(createFunctionIdentifier(functionModel), functionTypeBuilder.build());
      });
      if (!functionsMap.isEmpty()) {
        ModuleDefinition.Builder module = functionBindings.module(createModuleIdentifier(extensionModel));
        functionsMap.forEach(module::addElement);
      }
    });
  }


  private String createFunctionIdentifier(FunctionModel functionModel) {
    return createFunctionIdentifier(functionModel.getName());
  }

  private String createFunctionIdentifier(String functionName) {
    return functionName;
  }

  private String createModuleIdentifier(ExtensionModel extensionModel) {
    final ModuleNamespace moduleNamespace = new ModuleNamespace(extensionModel.getXmlDslModel().getPrefix());
    return moduleNamespace.getElements()[0];
  }

}
