package org.mule.datasense.impl.phases.typing.resolver;

import org.mule.datasense.impl.model.annotations.DefinesTypeAnnotation;
import org.mule.datasense.impl.model.annotations.OperationCallAnnotation;
import org.mule.datasense.impl.model.annotations.ThrowsErrorsTypeAnnotation;
import org.mule.datasense.impl.model.annotations.UsesTypeAnnotation;
import org.mule.datasense.impl.model.annotations.VoidOperationAnnotation;
import org.mule.datasense.impl.model.ast.AstNodeLocation;
import org.mule.datasense.impl.model.ast.AstNotification;
import org.mule.datasense.impl.model.ast.MessageProcessorNode;
import org.mule.datasense.impl.model.operation.InputArgument;
import org.mule.datasense.impl.model.operation.InputParameter;
import org.mule.datasense.impl.model.operation.OperationCall;
import org.mule.datasense.impl.model.reporting.NotificationMessages;
import org.mule.datasense.impl.model.types.EventType;
import org.mule.datasense.impl.model.types.TypeUtils;
import org.mule.datasense.impl.model.types.TypesHelper;
import org.mule.datasense.impl.phases.typing.TypingMuleAstVisitor;
import org.mule.datasense.impl.phases.typing.TypingMuleAstVisitorContext;
import org.mule.datasense.impl.phases.typing.resolver.errorhandling.ErrorMappingUtils;
import org.mule.datasense.impl.util.ExpressionLanguageUtils;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.message.api.MessageMetadataType;
import org.mule.metadata.message.api.MuleEventMetadataType;
import org.mule.metadata.message.api.MuleEventMetadataTypeBuilder;
import org.mule.metadata.message.api.el.ExpressionLanguageMetadataTypeResolver;
import org.mule.runtime.api.meta.model.error.ErrorModel;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

public class OperationCallTypeResolver extends SingleNodeTypeResolver {

  private final TargetProcessingSupport targetProcessingSupport = new TargetProcessingSupport();

  protected ExpressionLanguageMetadataTypeResolver.MessageCallback createMessageCallback(AstNotification astNotification,
                                                                                         AstNodeLocation astNodeLocation) {
    return new ExpressionLanguageMetadataTypeResolver.MessageCallback() {

      @Override
      public void warning(String message, ExpressionLanguageMetadataTypeResolver.MessageLocation messageLocation) {
        astNotification.reportWarning(astNodeLocation, NotificationMessages.MSG_SCRIPTING_LANGUAGE_WARNING("", message));
      }

      @Override
      public void error(String message, ExpressionLanguageMetadataTypeResolver.MessageLocation messageLocation) {
        astNotification.reportError(astNodeLocation, NotificationMessages.MSG_SCRIPTING_LANGUAGE_ERROR("", message));
      }
    };
  }

  @Override
  protected EventType resolve(MessageProcessorNode messageProcessorNode, EventType inputEventType,
                              TypingMuleAstVisitor typingMuleAstVisitor,
                              TypingMuleAstVisitorContext visitorContext) {
    OperationCallAnnotation operationCallAnnotation = messageProcessorNode.getAnnotation(OperationCallAnnotation.class)
        .orElseThrow(() -> new RuntimeException("Operation call annotation not present"));
    OperationCall operationCall = operationCallAnnotation.getOperationCall();

    MuleEventMetadataTypeBuilder inferredMuleEventTypeBuilder = TypesHelper.getMuleEventMetadataTypeBuilder();

    final ExpressionLanguageMetadataTypeResolver expressionLanguageMetadataTypeResolver =
        visitorContext.getExpressionLanguageMetadataTypeResolver();
    final ExpressionLanguageMetadataTypeResolver.MessageCallback messageCallback =
        createMessageCallback(visitorContext.getAstNotification(),
                              messageProcessorNode.getAstNodeLocation());

    operationCall.getInputMappings().forEach(inputMapping -> {
      InputArgument inputArgument = inputMapping.getInputArgument();
      InputParameter inputParameter = inputMapping.getInputParameter();
      if (inputArgument.getExpression() != null) {
        ExpressionLanguageUtils
            .resolveInputEventType(inputArgument.getExpression(), expressionLanguageMetadataTypeResolver,
                                   inputParameter.getMetadataType(),
                                   inferredMuleEventTypeBuilder,
                                   messageCallback);

      }
    });

    MuleEventMetadataType inferredInputMuleEventType = inferredMuleEventTypeBuilder.build();
    messageProcessorNode.annotate(new UsesTypeAnnotation(TypeUtils.asEventType(inferredInputMuleEventType)));

    messageProcessorNode.annotate(new ThrowsErrorsTypeAnnotation(resolveThrownErrors(operationCall, messageProcessorNode)));

    EventType outputEventType = inputEventType;
    if (!messageProcessorNode.isAnnotatedWith(VoidOperationAnnotation.class)) {
      MetadataType returnType = operationCall.getReturnType();

      if (returnType instanceof MessageMetadataType) {

        final MessageMetadataType returnMessageMetadataType = (MessageMetadataType) returnType;

        outputEventType = targetProcessingSupport
            .processTarget(operationCall.getTarget().orElse(null), operationCall.getTargetValueExpression(),
                           TypesHelper.getMuleEventMetadataTypeBuilder()
                               .message(returnMessageMetadataType).build(),
                           visitorContext, messageCallback);

        messageProcessorNode.annotate(new DefinesTypeAnnotation(outputEventType));
      }
    }
    return outputEventType;
  }

  private Set<ErrorModel> resolveThrownErrors(OperationCall operationCall, MessageProcessorNode messageProcessorNode) {
    final Set<ErrorModel> errorModels = operationCall.getErrorModels();
    return ErrorMappingUtils.createErrorMapper(messageProcessorNode).map(errorMapper -> {
      return errorModels.stream().map(errorMapper::map).collect(Collectors.toSet());
    }).orElse(errorModels);
  }

  @Override
  protected boolean isPropagates(MessageProcessorNode messageProcessorNode) {
    return true;
  }
}
