package org.mule.datasense.impl.phases.typing.resolver.errorhandling;

import org.mule.datasense.impl.model.types.TypesHelper;
import org.mule.datasense.impl.phases.typing.resolver.GlobalBindingMetadataTypes;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.ObjectTypeBuilder;
import org.mule.metadata.api.model.MetadataType;
import org.mule.runtime.api.meta.model.error.ErrorModel;
import org.mule.runtime.api.meta.model.error.ErrorModelBuilder;

import javax.xml.namespace.QName;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import static java.util.Collections.EMPTY_SET;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.mule.datasense.impl.model.types.TypesHelper.getTypeBuilder;

public class ErrorHandlingUtils {

  private static final String ERROR_CLASS = "org.mule.runtime.api.message.Error";

  public static Optional<ErrorModel> createErrorModel(String error) {
    return createErrorCode(error)
        .map(errorCode -> ErrorModelBuilder.newError(errorCode.getLocalPart(), errorCode.getNamespaceURI()).build());
  }

  public static Optional<QName> createErrorCode(String errorCode) {
    if (errorCode == null || errorCode.isEmpty()) {
      return empty();
    }

    errorCode = errorCode.trim();

    String namespace = null;
    String name = null;

    final String[] split = errorCode.split(":");
    switch (split.length) {
      case 1:
        name = split[0];
        break;
      case 2:
        namespace = split[0];
        name = split[1];
        break;
    }

    if (name != null) {
      return of(namespace != null ? new QName(namespace, name) : new QName(name));
    } else {
      return empty();
    }
  }

  public static List<QName> getErrorCodes(ErrorModel unhandledError) {
    List<QName> catchableErrorCodes = new ArrayList<>();
    ErrorModel currentErrorModel = unhandledError;
    while (currentErrorModel != null) {
      catchableErrorCodes.add(new QName(currentErrorModel.getNamespace(), currentErrorModel.getType()));
      currentErrorModel = currentErrorModel.getParent().orElse(null);
    }
    return catchableErrorCodes;
  }

  public static ErrorMatcher createErrorMatcher(String errorMatchExpression) {
    String[] errorCodes = errorMatchExpression.split(",");
    List<ErrorMatcher> errorMatchers =
        Arrays.stream(errorCodes).map(errorCode -> new DefaultErrorMatcher(errorCode.trim())).collect(Collectors.toList());
    return new CompositeErrorMatcher(errorMatchers);
  }


  private static Set<QName> getErrorCodes(Set<ErrorModel> errorModels) {
    Set<QName> result = new LinkedHashSet<>();
    errorModels.forEach(errorModel -> {
      if (errorModel.isHandleable()) {
        result.addAll(getErrorCodes(errorModel));
      }
    });
    return result;
  }

  public static MetadataType errorType() {
    return TypesHelper.getTypeFromJavaClass(ERROR_CLASS, GlobalBindingMetadataTypes.class.getClassLoader())
        .orElse(errorType(EMPTY_SET));
  }

  public static MetadataType errorType(Set<ErrorModel> unhandledErrors) {
    final BaseTypeBuilder errorTypeBuilder = getTypeBuilder();

    final Set<QName> errorCodes = getErrorCodes(unhandledErrors);

    List<String> namespaces = new ArrayList<>();
    List<String> identifiers = new ArrayList<>();
    errorCodes.forEach(errorCode -> {
      namespaces.add(errorCode.getNamespaceURI());
      identifiers.add(errorCode.getLocalPart());
    });

    final ObjectTypeBuilder typeBuilder = errorTypeBuilder.objectType();
    typeBuilder.addField().key("description").value().stringType();
    typeBuilder.addField().key("detailedDescription").value().stringType();
    final ObjectTypeBuilder errorType = typeBuilder.addField().key("errorType").value().objectType();
    errorType.addField().key("namespace").value().stringType().enumOf(namespaces.toArray(new String[namespaces.size()]));
    errorType.addField().key("identifier").value().stringType().enumOf(identifiers.toArray(new String[identifiers.size()]));

    return errorTypeBuilder.build();
  }
}
