package org.mule.datasense.impl.phases.typing.resolver;

import org.mule.datasense.impl.model.annotations.DefinesTypeAnnotation;
import org.mule.datasense.impl.model.annotations.ExpectedEventAnnotation;
import org.mule.datasense.impl.model.annotations.ExpectedInputAnnotation;
import org.mule.datasense.impl.model.annotations.IncomingEventAnnotation;
import org.mule.datasense.impl.model.annotations.MessageProcessorTypeDeclarationAnnotation;
import org.mule.datasense.impl.model.annotations.ResultEventAnnotation;
import org.mule.datasense.impl.model.annotations.ThrowsErrorsTypeAnnotation;
import org.mule.datasense.impl.model.annotations.UsesTypeAnnotation;
import org.mule.datasense.impl.model.ast.AstNodeLocation;
import org.mule.datasense.impl.model.ast.AstNotification;
import org.mule.datasense.impl.model.ast.MessageProcessorNode;
import org.mule.datasense.impl.model.types.EventType;
import org.mule.datasense.impl.model.types.TypeUtils;
import org.mule.datasense.impl.phases.builder.ComponentModelType;
import org.mule.datasense.impl.phases.builder.MuleAstParseProvider;
import org.mule.datasense.impl.phases.scoping.ExpectedAstVisitor;
import org.mule.datasense.impl.phases.scoping.ExpectedAstVisitorContext;
import org.mule.datasense.impl.phases.scoping.IncomingAstVisitor;
import org.mule.datasense.impl.phases.scoping.IncomingAstVisitorContext;
import org.mule.datasense.impl.phases.typing.TypingMuleAstVisitor;
import org.mule.datasense.impl.phases.typing.TypingMuleAstVisitorContext;
import org.mule.datasense.impl.util.AstMessageCallbackFactory;
import org.mule.datasense.impl.util.MessageCallbackFactory;
import org.mule.datasense.impl.util.MutableHolder;
import org.mule.metadata.api.model.FunctionParameter;
import org.mule.metadata.api.model.FunctionType;
import org.mule.metadata.message.api.MuleEventMetadataType;
import org.mule.metadata.message.api.el.ExpressionLanguageMetadataTypeResolver;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import static org.mule.datasense.impl.model.types.TypeUtils.USE_EVENT_PREDICATE;

public abstract class BaseTypeResolver implements TypeResolver {

  public Optional<ComponentModelType> getComponentModelType() {
    return Optional.empty();
  }

  @Override
  public final EventType resolveTypes(MessageProcessorNode messageProcessorNode, TypingMuleAstVisitor typingMuleAstVisitor,
                                      TypingMuleAstVisitorContext typingMuleAstVisitorContext) {
    if (definesErrorHandlingContext()) {
      typingMuleAstVisitorContext.getErrorHandlingEnvironment().pushContext("node-" + messageProcessorNode.getName());
    }

    EventType argumentEventType = typingMuleAstVisitorContext.getTypingEnvironment().get().resolveInput();

    final EventType declaredInputEventType = messageProcessorNode.getAnnotation(MessageProcessorTypeDeclarationAnnotation.class)
        .flatMap(messageProcessorTypeDeclarationAnnotation -> {
          final FunctionType functionType =
              messageProcessorTypeDeclarationAnnotation.getMessageProcessorTypeDeclaration().getFunctionType();
          return functionType.getParameters().stream().map(FunctionParameter::getType)
              .filter(metadataType -> metadataType instanceof MuleEventMetadataType)
              .map(metadataType -> TypeUtils.asEventType((MuleEventMetadataType) metadataType)).findFirst();
        }).orElse(new EventType());

    EventType retypedArgumentEventType = TypeUtils.merge(argumentEventType, declaredInputEventType);
    EventType resultEventType =
        resolve(messageProcessorNode, retypedArgumentEventType, typingMuleAstVisitor, typingMuleAstVisitorContext);

    messageProcessorNode.getAnnotation(ThrowsErrorsTypeAnnotation.class).ifPresent(throwsErrorsTypeAnnotation -> {
      typingMuleAstVisitorContext.getErrorHandlingEnvironment().throwErrors(throwsErrorsTypeAnnotation.getErrors());
    });

    messageProcessorNode.getAnnotation(MessageProcessorTypeDeclarationAnnotation.class)
        .ifPresent(messageProcessorTypeDeclarationAnnotation -> {
          final FunctionType functionType =
              messageProcessorTypeDeclarationAnnotation.getMessageProcessorTypeDeclaration().getFunctionType();
          functionType.getParameters().stream().map(FunctionParameter::getType)
              .filter(metadataType -> metadataType instanceof MuleEventMetadataType)
              .map(metadataType -> TypeUtils.asEventType((MuleEventMetadataType) metadataType)).findFirst()
              .ifPresent(eventType -> {
                final UsesTypeAnnotation usesTypeAnnotation =
                    messageProcessorNode.getOrCreateAnnotation(UsesTypeAnnotation.class,
                                                               () -> new UsesTypeAnnotation(new EventType()));
                usesTypeAnnotation.overrideWith(eventType);

              });
          functionType.getReturnType().filter(metadataType -> metadataType instanceof MuleEventMetadataType)
              .map(metadataType -> TypeUtils.asEventType((MuleEventMetadataType) metadataType)).ifPresent(eventType -> {
                final DefinesTypeAnnotation definesTypeAnnotation =
                    messageProcessorNode.getOrCreateAnnotation(DefinesTypeAnnotation.class,
                                                               () -> new DefinesTypeAnnotation(new EventType()));
                definesTypeAnnotation.overrideWith(eventType);
              });
        });;

    Optional<EventType> usesEventType = messageProcessorNode.getAnnotation(UsesTypeAnnotation.class)
        .map(UsesTypeAnnotation::getUsesEventType);
    Optional<EventType> definesEventType = messageProcessorNode.getAnnotation(DefinesTypeAnnotation.class)
        .map(DefinesTypeAnnotation::getDefinesEventType);

    EventType outputEventType = new EventType();
    if (isPropagates(messageProcessorNode)) {
      outputEventType = TypeUtils.merge(outputEventType, argumentEventType);
    }
    if (!isScope()) {
      outputEventType = TypeUtils.merge(outputEventType, resultEventType);
    }
    if (definesEventType.isPresent()) {
      outputEventType = TypeUtils.merge(outputEventType, definesEventType.get());
    }

    if (definesErrorHandlingContext()) {
      typingMuleAstVisitorContext.getErrorHandlingEnvironment().popContext();
    }

    return outputEventType;
  }

  protected boolean definesErrorHandlingContext() {
    return true;
  }

  protected EventType unifyEventTypes(EventType sourceEventType, EventType targetEventType, boolean source) {
    return source ? sourceEventType : targetEventType;
  }

  protected abstract EventType resolve(MessageProcessorNode messageProcessorNode, EventType inputEventType,
                                       TypingMuleAstVisitor typingMuleAstVisitor,
                                       TypingMuleAstVisitorContext visitorContext);

  protected abstract boolean isPropagates(MessageProcessorNode messageProcessorNode);

  @Override
  public EventType generateIncoming(MessageProcessorNode messageProcessorNode, IncomingAstVisitor incomingAstVisitor,
                                    IncomingAstVisitorContext visitorContext) {
    EventType incomingEventType = visitorContext.getIncomingEventType();
    IncomingEventAnnotation incomingEventAnnotation = new IncomingEventAnnotation(incomingEventType);
    messageProcessorNode.annotate(incomingEventAnnotation);

    /*
    visitorContext.logger()
        .debug(String.format("received event: %s in mp: %s", visitorContext.getIncomingEventType(),
                             messageProcessorNode.getName()));
    */

    final MutableHolder<EventType> propagatedEvent = new MutableHolder<>(incomingEventType);
    messageProcessorNode.getMessageProcessorNodes()
        .forEach(innerMessageProcessorNode -> {
          visitorContext.setIncomingEventType(propagatedEvent.getValue().orElse(new EventType()));
          final EventType childEventType = (EventType) innerMessageProcessorNode.accept(incomingAstVisitor, visitorContext);
          if (isSequential()) {
            propagatedEvent.setValue(childEventType);
          }
        });

    EventType declaredOutputEventType =
        messageProcessorNode.getAnnotation(DefinesTypeAnnotation.class).map(DefinesTypeAnnotation::getDefinesEventType)
            .orElse(new EventType());

    //    declaredOutputEventType = TypeUtils
    //        .annotate(declaredOutputEventType, new VarDeclMetadata(messageProcessorNode.getComponentModel().getComponentLocation()));

    EventType outputEventType = new EventType();
    if (isPropagates(messageProcessorNode)) {
      outputEventType = TypeUtils.merge(outputEventType, incomingEventType);
    }
    if (!isScope()) {
      outputEventType = TypeUtils.merge(outputEventType, propagatedEvent.getValue().orElse(new EventType()));
    }
    outputEventType = TypeUtils.merge(outputEventType, declaredOutputEventType);

    visitorContext.logger().exit(messageProcessorNode.getName());

    messageProcessorNode
        .annotate(new ResultEventAnnotation(outputEventType));

    return outputEventType;
  }

  protected EventType resolveInnerExpectedInputEventType(EventType expectedInputEventType,
                                                         MessageProcessorNode messageProcessorNode,
                                                         ExpectedAstVisitor expectedAstVisitor,
                                                         ExpectedAstVisitorContext visitorContext) {
    return expectedInputEventType;
  }

  protected EventType resolveInnerExpectedOutputEventType(EventType expectedOutputEventType,
                                                          EventType expectedOutputInnerEventType,
                                                          MessageProcessorNode messageProcessorNode,
                                                          ExpectedAstVisitor expectedAstVisitor,
                                                          ExpectedAstVisitorContext visitorContext) {
    return expectedOutputInnerEventType;
  }

  @Override
  public EventType generateExpected(MessageProcessorNode messageProcessorNode, ExpectedAstVisitor expectedAstVisitor,
                                    ExpectedAstVisitorContext visitorContext) {
    EventType expectedEventType = visitorContext.getExpectedEventType();
    ExpectedEventAnnotation expectedEventAnnotation = new ExpectedEventAnnotation(expectedEventType);
    messageProcessorNode.annotate(expectedEventAnnotation);

    EventType declaredOutputEventType = messageProcessorNode.getAnnotation(DefinesTypeAnnotation.class)
        .map(DefinesTypeAnnotation::getDefinesEventType).orElse(new EventType());
    if (isPropagates(messageProcessorNode)) {
      expectedEventType = TypeUtils.minus(expectedEventType, declaredOutputEventType);
    }

    EventType expectedOutputEventType =
        resolveInnerExpectedOutputEventType(visitorContext.getExpectedEventType(), expectedEventType, messageProcessorNode,
                                            expectedAstVisitor, visitorContext);

    final MutableHolder<EventType> propagatedEvent = new MutableHolder<>(isScope() ? new EventType() : expectedOutputEventType);
    if (isSequential()) {
      messageProcessorNode.getMessageProcessorNodes().collect(Collectors.toCollection(LinkedList::new)).descendingIterator()
          .forEachRemaining(innerMessageProcessorNode -> {
            visitorContext.setExpectedEventType(propagatedEvent.getValue().orElse(new EventType()));
            final EventType childEventType = (EventType) innerMessageProcessorNode.accept(expectedAstVisitor, visitorContext);
            propagatedEvent.setValue(childEventType);
          });
    } else {
      List<EventType> childEventTypes = new ArrayList<>();
      messageProcessorNode.getMessageProcessorNodes().forEach(childMessageProcessorNode -> {
        visitorContext.setExpectedEventType(propagatedEvent.getValue().orElse(new EventType()));
        final EventType childEventType = (EventType) childMessageProcessorNode.accept(expectedAstVisitor, visitorContext);
        childEventTypes.add(childEventType);
      });
      propagatedEvent.setValue(TypeUtils.intersection(childEventTypes));
    }

    /*
    EventType declaredOutputEventType = messageProcessorNode.getAnnotation(DefinesTypeAnnotation.class)
        .map(DefinesTypeAnnotation::getDefinesEventType).orElse(new EventType());
    EventType eventType = isScope() ? expectedEventType : new EventType(propagatedEvent.getValue().getVarDecls()
        .filter(varDecl -> !declaredOutputEventType.get(varDecl.getName()).isPresent()));
    EventType declaredInputEventType =
        resolveExpectedInputEventType(propagatedEvent.getValue(), messageProcessorNode, expectedAstVisitor, visitorContext);
    EventType inputEventType =
        isPropagates(messageProcessorNode) ? TypeUtils.merge(eventType, declaredInputEventType) : declaredInputEventType;
    */

    EventType inputEventType = new EventType();
    if (isPropagates(messageProcessorNode)) {
      inputEventType = TypeUtils.merge(inputEventType, expectedEventType);
    }
    //    if (!isScope()) {
    inputEventType = TypeUtils.merge(inputEventType, propagatedEvent.getValue().orElse(new EventType()));
    //    }
    EventType declaredInputEventType =
        messageProcessorNode.getAnnotation(UsesTypeAnnotation.class).map(UsesTypeAnnotation::getUsesEventType)
            .orElse(resolveExpectedInputEventType(propagatedEvent.getValue().orElse(new EventType()), messageProcessorNode,
                                                  expectedAstVisitor, visitorContext));
    inputEventType = TypeUtils.merge(inputEventType, declaredInputEventType);

    inputEventType = TypeUtils.filter(inputEventType, USE_EVENT_PREDICATE);

    messageProcessorNode
        .annotate(new ExpectedInputAnnotation(inputEventType));

    visitorContext.logger().exit(messageProcessorNode.getName());
    return inputEventType;
  }

  protected EventType resolveExpectedInputEventType(EventType innerEventType, MessageProcessorNode messageProcessorNode,
                                                    ExpectedAstVisitor expectedAstVisitor,
                                                    ExpectedAstVisitorContext visitorContext) {
    return new EventType();
  }

  @Override
  public Optional<MuleAstParseProvider> getParseProvider() {
    return Optional.empty();
  }

  protected boolean isScope() {
    return false;
  }

  protected boolean isSequential() {
    return true;
  }

  protected ExpressionLanguageMetadataTypeResolver.MessageCallback createMessageCallback(AstNotification astNotification,
                                                                                         AstNodeLocation astNodeLocation,
                                                                                         String expression) {
    return createMessageCallbackFactory(astNotification, astNodeLocation).createMessageCallback(expression);
  }

  protected MessageCallbackFactory createMessageCallbackFactory(AstNotification astNotification,
                                                                AstNodeLocation astNodeLocation) {
    return new AstMessageCallbackFactory(astNotification, astNodeLocation);
  }
}
