package org.mule.datasense.impl.phases.typing.resolver;

import org.mule.datasense.impl.model.types.EventType;
import org.mule.datasense.impl.model.types.TypeUtils;
import org.mule.datasense.impl.model.types.TypesHelper;
import org.mule.datasense.impl.model.types.VarDecl;
import org.mule.datasense.impl.phases.typing.TypingMuleAstVisitorContext;
import org.mule.datasense.impl.util.MessageCallbackFactory;
import org.mule.metadata.MetadataFormats;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.message.api.MuleEventMetadataType;
import org.mule.metadata.message.api.el.ExpressionLanguageMetadataTypeResolver;

import java.util.stream.Stream;

import static org.mule.datasense.impl.model.types.TypeUtils.asMessageMetadataTypeOrEmptyMessage;
import static org.mule.datasense.impl.model.types.TypeUtils.merge;
import static org.mule.datasense.impl.model.types.TypesHelper.MULE_EVENT_PAYLOAD;
import static org.mule.datasense.impl.util.ExpressionLanguageUtils.extractExpression;
import static org.mule.datasense.impl.util.ExpressionLanguageUtils.resolveExpressionType;

public class TargetProcessingSupport {

  private MetadataType resolveTargetMetadataType(MuleEventMetadataType muleEventMetadataType, String targetValue, String mimeType,
                                                 TypingMuleAstVisitorContext visitorContext,
                                                 MessageCallbackFactory messageCallbackFactory) {
    final ExpressionLanguageMetadataTypeResolver expressionLanguageMetadataTypeResolver =
        visitorContext.getExpressionLanguageMetadataTypeResolver();

    return extractExpression(targetValue).map(targetValueExpression -> {
      return resolveExpressionType(targetValueExpression,
                                   mimeType,
                                   muleEventMetadataType,
                                   visitorContext.getTypeBindings(),
                                   expressionLanguageMetadataTypeResolver,
                                   messageCallbackFactory.createMessageCallback(targetValueExpression));
    }).orElse(asMessageMetadataTypeOrEmptyMessage(muleEventMetadataType.getMessageType()));
  }

  private EventType resolveTargetEventType(String target, String targetValue, String mimeType,
                                           MuleEventMetadataType muleEventMetadataType,
                                           TypingMuleAstVisitorContext visitorContext,
                                           MessageCallbackFactory messageCallbackFactory) {
    MetadataType targetMetadataType =
        resolveTargetMetadataType(muleEventMetadataType, targetValue, mimeType, visitorContext, messageCallbackFactory);
    return new EventType(Stream.of(new VarDecl(target, targetMetadataType)));
  }

  private EventType resolveVariablesEventType(MuleEventMetadataType muleEventMetadataType) {
    return new EventType(muleEventMetadataType.getVariables().getFields().stream().map(
                                                                                       objectFieldType -> new VarDecl(objectFieldType
                                                                                           .getKey().getName()
                                                                                           .getLocalPart(), objectFieldType
                                                                                               .getValue())));
  }

  private EventType resolveAttributesEventType(MuleEventMetadataType muleEventMetadataType) {
    return muleEventMetadataType.getMessageType().getAttributesType().map(attributesType -> {
      return new EventType(Stream.of(new VarDecl(TypesHelper.MULE_EVENT_ATTRIBUTES, attributesType)));
    }).orElse(new EventType());
  }

  public EventType processTarget(String target, String targetValue, String mimeType, MuleEventMetadataType muleEventMetadataType,
                                 TypingMuleAstVisitorContext visitorContext,
                                 MessageCallbackFactory messageCallbackFactory) {
    if (target == null && mimeType == null) {
      return TypeUtils.asEventType(muleEventMetadataType);
    }

    boolean includeAttributes = false;
    if (target == null) {
      target = MULE_EVENT_PAYLOAD;
      targetValue = "#[payload]";
      includeAttributes = true;
    }
    if (mimeType == null) {
      mimeType = MetadataFormats.JAVA.getValidMimeTypes().stream().findFirst().orElse("application/java");
    }
    EventType eventType = resolveVariablesEventType(muleEventMetadataType);
    if (includeAttributes) {
      eventType = merge(eventType, resolveAttributesEventType(muleEventMetadataType));
    }
    return merge(eventType, resolveTargetEventType(target, targetValue, mimeType, muleEventMetadataType, visitorContext,
                                                   messageCallbackFactory));
  }

  public EventType processTarget(String target, String targetValue, MuleEventMetadataType muleEventMetadataType,
                                 TypingMuleAstVisitorContext visitorContext,
                                 MessageCallbackFactory messageCallbackFactory) {
    return processTarget(target, targetValue, null, muleEventMetadataType, visitorContext, messageCallbackFactory);
  }

  public EventType processTarget(String target, String targetValue, String mimeType, EventType eventType,
                                 TypingMuleAstVisitorContext visitorContext,
                                 MessageCallbackFactory messageCallbackFactory) {
    MuleEventMetadataType muleEventMetadataType = TypeUtils.asMuleEventMetadataType(eventType).build();
    return processTarget(target, targetValue, mimeType, muleEventMetadataType, visitorContext, messageCallbackFactory);
  }

  public EventType processTarget(String target, String targetValue, EventType eventType,
                                 TypingMuleAstVisitorContext visitorContext,
                                 MessageCallbackFactory messageCallbackFactory) {
    return processTarget(target, targetValue, null, eventType, visitorContext, messageCallbackFactory);
  }
}
