package org.mule.datasense.impl.model.ast;

import org.mule.datasense.impl.model.annotations.FunctionBindingsAnnotation;
import org.mule.datasense.impl.model.annotations.GlobalBindingsAnnotation;
import org.mule.datasense.impl.model.annotations.MuleApplicationAnnotation;
import org.mule.datasense.impl.phases.typing.resolver.TypeResolverRegistry;
import org.mule.datasense.impl.util.AnnotationSupport;
import org.mule.datasense.impl.util.TreeLogSupport;
import org.mule.metadata.message.api.el.ExpressionLanguageMetadataTypeResolver;
import org.mule.metadata.message.api.el.TypeBindings;

import java.util.Optional;
import java.util.Set;
import java.util.function.Supplier;

public abstract class BaseAstNodeVisitorContext {

  private final TreeLogSupport treeLogSupport;
  private final AstNotification astNotification;
  private AnnotationSupport<VisitorContextAnnotation> annotationSupport;
  private TypeResolverRegistry typeResolverRegistry;
  private ExpressionLanguageMetadataTypeResolver expressionLanguageMetadataTypeResolver;

  public BaseAstNodeVisitorContext(AstNotification astNotification) {
    this(null, astNotification);
  }

  public BaseAstNodeVisitorContext(TypeResolverRegistry typeResolverRegistry, AstNotification astNotification) {
    this.typeResolverRegistry = typeResolverRegistry;
    this.astNotification = astNotification;
    treeLogSupport = new TreeLogSupport();
    annotationSupport = new AnnotationSupport<>();
    this.expressionLanguageMetadataTypeResolver =
        ExpressionLanguageMetadataTypeResolver.getInstance(BaseAstNodeVisitorContext.class.getClassLoader());
  }

  public ExpressionLanguageMetadataTypeResolver getExpressionLanguageMetadataTypeResolver() {
    return expressionLanguageMetadataTypeResolver;
  }

  public TypeResolverRegistry getTypeResolverRegistry() {
    return typeResolverRegistry;
  }

  public AstNotification getAstNotification() {
    return astNotification;
  }

  public TreeLogSupport logger() {
    return treeLogSupport;
  }

  public Set<VisitorContextAnnotation> getAnnotations() {
    return annotationSupport.getAnnotations();
  }

  public <T extends VisitorContextAnnotation> Optional<T> getAnnotation(Class<T> annotationClass) {
    return annotationSupport.getAnnotation(annotationClass);
  }

  public <T extends VisitorContextAnnotation> void annotate(T annotation) {
    annotationSupport.annotate(annotation);
  }

  public <T extends VisitorContextAnnotation> Optional<T> deannotate(Class<T> annotation) {
    return annotationSupport.deannotate(annotation);
  }

  public <T extends VisitorContextAnnotation> boolean isAnnotatedWith(Class<T> annotationClass) {
    return annotationSupport.isAnnotatedWith(annotationClass);
  }

  public <T extends VisitorContextAnnotation> T getOrCreateAnnotation(Class<T> annotationClass,
                                                                      Supplier<T> supplier) {
    return annotationSupport.getOrCreateAnnotation(annotationClass, supplier);
  }

  public TypeBindings getTypeBindings() {
    final TypeBindings.Builder result = TypeBindings.builder();
    getAnnotation(MuleApplicationAnnotation.class).map(MuleApplicationAnnotation::getMuleApplicationNode)
        .ifPresent(muleApplicationNode -> {
          muleApplicationNode.getAnnotation(GlobalBindingsAnnotation.class).map(GlobalBindingsAnnotation::getTypeBindings)
              .ifPresent(
                         result::addAll);
          muleApplicationNode.getAnnotation(FunctionBindingsAnnotation.class).map(FunctionBindingsAnnotation::getFunctionBindings)
              .ifPresent(
                         result::addAll);
        });
    return result.build();
  }
}
