package org.mule.datasense.impl.phases.typing.resolver;

import org.mule.datasense.impl.model.annotations.DefinesTypeAnnotation;
import org.mule.datasense.impl.model.annotations.UsesTypeAnnotation;
import org.mule.datasense.impl.model.ast.MessageProcessorNode;
import org.mule.datasense.impl.model.types.EventType;
import org.mule.datasense.impl.model.types.TypeUtils;
import org.mule.datasense.impl.model.types.TypesHelper;
import org.mule.datasense.impl.phases.builder.ComponentModelType;
import org.mule.datasense.impl.phases.typing.TypingMuleAstVisitor;
import org.mule.datasense.impl.phases.typing.TypingMuleAstVisitorContext;
import org.mule.datasense.impl.util.ComponentIdentifierUtils;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.ObjectTypeBuilder;
import org.mule.metadata.api.builder.UnionTypeBuilder;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.message.api.MessageMetadataType;
import org.mule.metadata.message.api.MessageMetadataTypeBuilder;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.location.LocationPart;
import org.mule.runtime.config.internal.model.ComponentModel;
import org.mule.runtime.dsl.api.component.config.DefaultComponentLocation;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import static org.mule.datasense.impl.phases.builder.MuleAstParser.MULE_CORE;

public class ScatterGatherTypeResolver extends StructuralNodeTypeResolver {

  private static final ComponentIdentifier COLLECT_LIST =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE, "collect-list");

  private interface CollectStrategy {

    void processMessageProcessorOutput(MessageProcessorNode messageProcessorNode, EventType eventType);

    EventType build();
  }

  private static abstract class BaseCollectStrategy implements CollectStrategy {

    private MessageMetadataTypeBuilder messageMetadataTypeBuilder;

    public BaseCollectStrategy() {
      this.messageMetadataTypeBuilder = TypesHelper.getMessageMetadataTypeBuilder();
    }

    protected MessageMetadataTypeBuilder getMessageMetadataTypeBuilder() {
      return messageMetadataTypeBuilder;
    }

    protected void buildMessage() {}

    @Override
    public EventType build() {
      buildMessage();
      return new EventType(TypeUtils.asVarDecls(messageMetadataTypeBuilder.build()));
    }
  }

  private static class MapCollectStrategy extends BaseCollectStrategy {

    private ObjectTypeBuilder objectTypeBuilder;

    public MapCollectStrategy() {
      final MessageMetadataTypeBuilder messageMetadataTypeBuilder = getMessageMetadataTypeBuilder();
      objectTypeBuilder = messageMetadataTypeBuilder.payload().objectType();
      messageMetadataTypeBuilder.attributes().voidType();
    }

    @Override
    public void processMessageProcessorOutput(MessageProcessorNode messageProcessorNode, EventType eventType) {
      final MessageMetadataTypeBuilder routeMessageMetadataTypeBuilder = TypesHelper.getMessageMetadataTypeBuilder();
      final MetadataType routePayloadMetadataType = TypeUtils.getMessageMetadataType(eventType).flatMap(
                                                                                                        MessageMetadataType::getPayloadType)
          .orElse(TypesHelper.getTypeBuilder().anyType().build());
      TypeUtils.getMessageMetadataType(eventType).flatMap(
                                                          MessageMetadataType::getAttributesType)
          .ifPresent(routeMessageMetadataTypeBuilder::attributes);
      routeMessageMetadataTypeBuilder.payload(routePayloadMetadataType);

      final DefaultComponentLocation defaultComponentLocation =
          messageProcessorNode.getComponentModel().getComponentLocation();
      final LocationPart[] parts =
          defaultComponentLocation.getParts().toArray(new LocationPart[defaultComponentLocation.getParts().size()]);
      String routeId = buildRouteId(parts);
      objectTypeBuilder.addField().required().key(routeId).value(routeMessageMetadataTypeBuilder.build());
    }

    private String buildRouteId(LocationPart[] parts) {
      return parts[parts.length - 1].getPartPath();
    }
  }

  private static class ListCollectStrategy extends BaseCollectStrategy {

    private List<MessageMetadataTypeBuilder> messageMetadataTypeBuilders;

    public ListCollectStrategy() {
      final MessageMetadataTypeBuilder messageMetadataTypeBuilder = getMessageMetadataTypeBuilder();
      messageMetadataTypeBuilder.attributes().voidType();
      messageMetadataTypeBuilders = new ArrayList<>();
    }

    @Override
    public void processMessageProcessorOutput(MessageProcessorNode messageProcessorNode, EventType eventType) {
      final MessageMetadataTypeBuilder routeMessageMetadataTypeBuilder = TypesHelper.getMessageMetadataTypeBuilder();

      final MetadataType routePayloadMetadataType = TypeUtils.getMessageMetadataType(eventType).flatMap(
                                                                                                        MessageMetadataType::getPayloadType)
          .orElse(TypesHelper.getTypeBuilder().anyType().build());
      routeMessageMetadataTypeBuilder.payload(routePayloadMetadataType);

      TypeUtils.getMessageMetadataType(eventType).flatMap(
                                                          MessageMetadataType::getAttributesType)
          .ifPresent(routeMessageMetadataTypeBuilder::attributes);
      messageMetadataTypeBuilders.add(routeMessageMetadataTypeBuilder);
    }

    @Override
    protected void buildMessage() {
      final BaseTypeBuilder elementTypeBuilder = TypesHelper.getTypeBuilder();
      if (messageMetadataTypeBuilders.isEmpty()) {
        elementTypeBuilder.anyType();
      } else {
        final UnionTypeBuilder unionTypeBuilder = elementTypeBuilder.unionType();
        messageMetadataTypeBuilders.forEach(unionTypeBuilder::of);
      }
      final MessageMetadataTypeBuilder messageMetadataTypeBuilder = getMessageMetadataTypeBuilder();
      messageMetadataTypeBuilder.payload().arrayType().of(elementTypeBuilder);
    }
  }

  private boolean isCollectList(MessageProcessorNode scatterGatherMessageProcessorNode) {
    final ComponentModel scatterGatherComponentModel = scatterGatherMessageProcessorNode.getComponentModel();
    for (ComponentModel childComponentModel : scatterGatherComponentModel.getInnerComponents()) {
      if (COLLECT_LIST.equals(ComponentIdentifierUtils.createFromComponentModel(childComponentModel))) {
        return true;
      }
    }
    return false;
  }

  private CollectStrategy resolveCollectStrategy(MessageProcessorNode scatterGatherMessageProcessorNode) {
    return isCollectList(scatterGatherMessageProcessorNode) ? new ListCollectStrategy() : new MapCollectStrategy();
  }


  @Override
  protected EventType resolve(MessageProcessorNode messageProcessorNode, EventType inputEventType,
                              TypingMuleAstVisitor typingMuleAstVisitor,
                              TypingMuleAstVisitorContext typingMuleAstVisitorContext) {

    final CollectStrategy collectStrategy = resolveCollectStrategy(messageProcessorNode);
    List<EventType> nestedEventTypes = new ArrayList<>();
    messageProcessorNode.getMessageProcessorNodes().forEach(pipedMessageProcessorNode -> {
      final EventType nestedEventType =
          typingMuleAstVisitor.resolveType(pipedMessageProcessorNode, inputEventType, typingMuleAstVisitorContext);
      nestedEventTypes.add(nestedEventType);
      collectStrategy.processMessageProcessorOutput(pipedMessageProcessorNode, nestedEventType);
    });

    EventType outputEventType = TypeUtils.merge(
                                                TypeUtils.union(nestedEventTypes,
                                                                typingMuleAstVisitorContext
                                                                    .getExpressionLanguageMetadataTypeResolver()),
                                                collectStrategy.build());

    messageProcessorNode.annotate(new UsesTypeAnnotation(new EventType()));
    messageProcessorNode.annotate(new DefinesTypeAnnotation(outputEventType));

    return outputEventType;
  }

  @Override
  public Optional<ComponentModelType> getComponentModelType() {
    return Optional.of(ComponentModelType.MESSAGE_PROCESSOR_NODE);
  }

  @Override
  protected boolean isSequential() {
    return false;
  }

  @Override
  protected boolean isPropagates(MessageProcessorNode messageProcessorNode) {
    return true;
  }

  @Override
  protected boolean isScope() {
    return true;
  }

}
