package org.mule.datasense.impl.phases.typing.resolver.transform;

import org.mule.datasense.impl.model.ast.AstNodeLocation;
import org.mule.datasense.impl.model.ast.AstNotification;
import org.mule.datasense.impl.model.event.MuleEventExprBuilder;
import org.mule.datasense.impl.model.event.SimpleExprBuilder;
import org.mule.datasense.impl.model.event.ValueExprBuilder;
import org.mule.datasense.impl.model.reporting.NotificationMessages;
import org.mule.datasense.impl.util.ComponentIdentifierUtils;
import org.mule.runtime.config.internal.model.ComponentModel;import org.mule.runtime.api.component.ComponentIdentifier;;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URI;
import java.util.Optional;
import java.util.function.Function;

import static java.util.Optional.empty;
import static java.util.Optional.ofNullable;
import static org.apache.commons.io.FileUtils.readFileToString;
import static org.apache.commons.io.FileUtils.toFile;
import static org.mule.datasense.impl.phases.builder.MuleAstParser.MULE_EE_CORE;

public class TransformParser {

  static final ComponentIdentifier MESSAGE = ComponentIdentifierUtils.createFromNamespaceAndName(MULE_EE_CORE, "message");
  static final ComponentIdentifier VARIABLES = ComponentIdentifierUtils.createFromNamespaceAndName(MULE_EE_CORE, "variables");
  static final ComponentIdentifier SET_PAYLOAD = ComponentIdentifierUtils.createFromNamespaceAndName(MULE_EE_CORE, "set-payload");
  static final ComponentIdentifier SET_ATTRIBUTES =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_EE_CORE, "set-attributes");
  static final ComponentIdentifier SET_VARIABLE =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_EE_CORE, "set-variable");
  static final String ATTR_VARIABLE_NAME = "variableName";
  static final String ATTR_RESOURCE = "resource";

  private static class TransformParserContext {

    private final Function<String, Optional<URI>> resourceResolver;
    private final AstNotification astNotification;
    private final AstNodeLocation astNodeLocation;

    public TransformParserContext(Function<String, Optional<URI>> resourceResolver,
                                  AstNotification astNotification, AstNodeLocation astNodeLocation) {
      this.resourceResolver = resourceResolver;
      this.astNotification = astNotification;
      this.astNodeLocation = astNodeLocation;
    }

    public Optional<Function<String, Optional<URI>>> getResourceResolver() {
      return ofNullable(resourceResolver);
    }

    public Optional<AstNotification> getAstNotification() {
      return ofNullable(astNotification);
    }

    public Optional<AstNodeLocation> getAstNodeLocation() {
      return ofNullable(astNodeLocation);
    }
  }

  private Optional<String> findResourceExpression(String resource, TransformParserContext transformParserContext) {
    return transformParserContext.getResourceResolver().map(resourceResolver -> {
      return resourceResolver.apply(resource).map(uri -> {
        try {
          return readFileToString(toFile(uri.toURL()));
        } catch (Exception e) {
          transformParserContext.getAstNotification()
              .ifPresent(astNotification -> astNotification.reportError(transformParserContext.getAstNodeLocation().orElse(null),
                                                                        NotificationMessages.MSG_RESOURCE_NOT_FOUND(resource)));
          return null;
        }
      });
    }).orElse(empty());
  }

  private Optional<String> findExpression(ComponentModel componentModel, TransformParserContext transformParserContext) {
    return ofNullable(componentModel.getParameters().get(ATTR_RESOURCE))
        .map(resource -> findResourceExpression(resource, transformParserContext))
        .orElse(ofNullable(componentModel.getTextContent()));
  }

  public void parseMessage(ComponentModel componentModel, MuleEventExprBuilder muleEventExprBuilder,
                           TransformParserContext transformParserContext) {
    componentModel.getInnerComponents().forEach(childComponentModel -> {
      ComponentIdentifier componentIdentifier = ComponentIdentifierUtils.createFromComponentModel(childComponentModel);
      if (SET_PAYLOAD.equals(componentIdentifier)) {
        findExpression(childComponentModel, transformParserContext).ifPresent(expression -> {
          muleEventExprBuilder.message(m -> m.value(m1 -> m1.payload(new SimpleExprBuilder(expression))));
        });
      } else if (SET_ATTRIBUTES.equals(componentIdentifier)) {
        findExpression(childComponentModel, transformParserContext).ifPresent(expression -> {
          muleEventExprBuilder.message(m -> m.value(m1 -> m1.attributes(new SimpleExprBuilder(expression))));
        });
      }
    });
  }

  public void parseVariables(ComponentModel componentModel, MuleEventExprBuilder muleEventExprBuilder,
                             TransformParserContext transformParserContext) {
    componentModel.getInnerComponents().forEach(childComponentModel -> {
      ComponentIdentifier componentIdentifier = ComponentIdentifierUtils.createFromComponentModel(childComponentModel);
      if (SET_VARIABLE.equals(componentIdentifier)) {
        ofNullable(childComponentModel.getParameters().get(ATTR_VARIABLE_NAME)).ifPresent(name -> {
          findExpression(childComponentModel, transformParserContext).ifPresent(expression -> {
            muleEventExprBuilder.variable(name, v -> v.value(new ValueExprBuilder(new SimpleExprBuilder(expression))));
          });
        });
      }
    });
  }

  public MuleEventExprBuilder parse(ComponentModel transformComponentModel, TransformParserContext transformParserContext) {
    MuleEventExprBuilder muleEventExprBuilder = new MuleEventExprBuilder();
    transformComponentModel.getInnerComponents().forEach(componentModel -> {
      ComponentIdentifier componentIdentifier = ComponentIdentifierUtils.createFromComponentModel(componentModel);
      if (MESSAGE.equals(componentIdentifier)) {
        parseMessage(componentModel, muleEventExprBuilder, transformParserContext);
      } else if (VARIABLES.equals(componentIdentifier)) {
        parseVariables(componentModel, muleEventExprBuilder, transformParserContext);
      }
    });
    return muleEventExprBuilder;
  }

  public MuleEventExprBuilder parse(ComponentModel transformComponentModel, Function<String, Optional<URI>> resourceResolver,
                                    AstNotification astNotification, AstNodeLocation astNodeLocation) {
    return parse(transformComponentModel, new TransformParserContext(resourceResolver, astNotification, astNodeLocation));
  }

  public MuleEventExprBuilder parse(ComponentModel transformComponentModel) {
    return parse(transformComponentModel, null, null, null);
  }

}
