package org.mule.datasense.impl.phases.annotators;

import org.mule.datasense.impl.DefaultDataSense;
import org.mule.datasense.impl.model.annotations.OperationCallAnnotation;
import org.mule.datasense.impl.model.annotations.OperationCallBuilderAnnotation;
import org.mule.datasense.impl.model.annotations.TypeResolverAnnotation;
import org.mule.datasense.impl.model.annotations.VoidOperationAnnotation;
import org.mule.datasense.impl.model.ast.MessageProcessorNode;
import org.mule.datasense.impl.model.operation.OperationCall;
import org.mule.datasense.impl.model.operation.OperationCallBuilder;
import org.mule.datasense.impl.phases.typing.AnnotatingMuleAstVisitorContext;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.VoidType;
import org.mule.metadata.message.api.MessageMetadataType;
import org.mule.runtime.api.component.ComponentIdentifier;

import java.util.List;

import static java.util.Arrays.asList;
import static org.mule.datasense.impl.DefaultDataSense.COMPONENT_IDENTIFIER_SET_PAYLOAD;
import static org.mule.datasense.impl.DefaultDataSense.COMPONENT_IDENTIFIER_SET_PAYLOAD_ATTRIBUTES;
import static org.mule.datasense.impl.DefaultDataSense.COMPONENT_IDENTIFIER_SET_VARIABLE;

public class OperationCallAnnotator extends BaseAnnotator {

  private static final List<ComponentIdentifier> whiteListedOperations =
      asList(COMPONENT_IDENTIFIER_SET_PAYLOAD, COMPONENT_IDENTIFIER_SET_VARIABLE, COMPONENT_IDENTIFIER_SET_PAYLOAD_ATTRIBUTES);

  private static boolean isWhiteListedOperation(MessageProcessorNode messageProcessorNode) {
    return whiteListedOperations.contains(messageProcessorNode.getComponentIdentifier());
  }

  private final SourceAnnotator sourceAnnotator;
  private final OperationAnnotator operationAnnotator;

  public OperationCallAnnotator() {
    sourceAnnotator = new SourceAnnotator();
    operationAnnotator = new OperationAnnotator();
  }

  @Override
  public void annotate(MessageProcessorNode messageProcessorNode,
                       AnnotatingMuleAstVisitorContext annotatingMuleAstVisitorContext) {
    if (!messageProcessorNode.isRootMessageProcessorNode()) {
      if (!messageProcessorNode.getAnnotation(TypeResolverAnnotation.class).isPresent()) {
        annotateOperationCall(messageProcessorNode, annotatingMuleAstVisitorContext,
                              DefaultDataSense.COMPONENT_IDENTIFIER_OPERATION_CALL);
      } else {
        if (isWhiteListedOperation(messageProcessorNode)) {
          annotateOperationCall(messageProcessorNode, annotatingMuleAstVisitorContext);
        }
      }
    }
  }

  private void annotateOperationCall(MessageProcessorNode messageProcessorNode,
                                     AnnotatingMuleAstVisitorContext annotatingMuleAstVisitorContext,
                                     ComponentIdentifier typeResolver) {
    messageProcessorNode.getComponentModelType().ifPresent(componentModelType -> {
      switch (componentModelType) {
        case MESSAGE_PROCESSOR_NODE:
          operationAnnotator.annotate(messageProcessorNode, annotatingMuleAstVisitorContext);
          break;
        case MESSAGE_SOURCE_NODE:
          sourceAnnotator.annotate(messageProcessorNode, annotatingMuleAstVisitorContext);
          break;
      }
    });

    messageProcessorNode.getAnnotation(OperationCallBuilderAnnotation.class).ifPresent(operationCallBuilderAnnotation -> {
      OperationCallBuilder operationCallBuilder = operationCallBuilderAnnotation.getOperationCallBuilder();
      OperationCall operationCall = operationCallBuilder.build();
      if (isVoidOperation(operationCall)) {
        messageProcessorNode.annotate(new VoidOperationAnnotation());
      }
      messageProcessorNode.annotate(new OperationCallAnnotation(operationCall));
      if (typeResolver != null) {
        messageProcessorNode.annotate(new TypeResolverAnnotation(typeResolver));
      }
    });
  }

  private void annotateOperationCall(MessageProcessorNode messageProcessorNode,
                                     AnnotatingMuleAstVisitorContext annotatingMuleAstVisitorContext) {
    annotateOperationCall(messageProcessorNode, annotatingMuleAstVisitorContext, null);
  }

  private boolean isVoidOperation(OperationCall operationCall) {
    boolean result = false;
    final MetadataType returnType = operationCall.getReturnType();
    if (returnType instanceof MessageMetadataType) {
      MessageMetadataType messageMetadataType = (MessageMetadataType) returnType;
      result =
          (!messageMetadataType.getPayloadType().isPresent() || messageMetadataType.getPayloadType().get() instanceof VoidType)
              && (!messageMetadataType.getAttributesType().isPresent()
                  || messageMetadataType.getAttributesType().get() instanceof VoidType);
    }
    return result;
  }

}
