package org.mule.datasense.impl.phases.typing;

import org.mule.datasense.impl.DataSenseProviderResolver;
import org.mule.datasense.impl.model.annotations.ComponentLocationAnnotation;
import org.mule.datasense.impl.model.annotations.DefinesTypeAnnotation;
import org.mule.datasense.impl.model.annotations.MessageProcessorTypeDeclarationAnnotation;
import org.mule.datasense.impl.model.annotations.MessageSourceRegionContextAnnotation;
import org.mule.datasense.impl.model.annotations.MuleApplicationAnnotation;
import org.mule.datasense.impl.model.annotations.MuleFlowAnnotation;
import org.mule.datasense.impl.model.annotations.UsesTypeAnnotation;
import org.mule.datasense.impl.model.ast.AstNode;
import org.mule.datasense.impl.model.ast.AstNodeVisitor;
import org.mule.datasense.impl.model.ast.AstNotification;
import org.mule.datasense.impl.model.ast.MessageProcessorNode;
import org.mule.datasense.impl.model.ast.MuleApplicationNode;
import org.mule.datasense.impl.model.ast.MuleFlowNode;
import org.mule.datasense.impl.model.types.TypeUtils;
import org.mule.datasense.impl.phases.annotators.AnnotatorsRegistry;
import org.mule.metadata.api.model.FunctionParameter;
import org.mule.metadata.api.model.FunctionType;
import org.mule.metadata.message.api.MuleEventMetadataType;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.stream.Collectors;

public class AnnotatingMuleAstVisitor implements AstNodeVisitor<AnnotatingMuleAstVisitorContext> {

  private final AnnotatorsRegistry annotatorsRegistry;

  public AnnotatingMuleAstVisitor(AnnotatorsRegistry annotatorsRegistry) {
    this.annotatorsRegistry = annotatorsRegistry;
  }

  public AnnotatorsRegistry getAnnotatorsRegistry() {
    return annotatorsRegistry;
  }

  @Override
  public Object visit(MessageProcessorNode messageProcessorNode, AnnotatingMuleAstVisitorContext visitorContext) {
    visitorContext.logger().enter(messageProcessorNode.getName());

    getAnnotatorsRegistry().annotate(messageProcessorNode, visitorContext);

    boolean root = messageProcessorNode.isRootMessageProcessorNode();
    if (root) {
      Iterator<MessageProcessorNode> messageProcessorNodeIterator =
          messageProcessorNode.getMessageProcessorNodes().collect(Collectors.toList()).iterator();
      if (messageProcessorNodeIterator.hasNext()) {
        MessageProcessorNode child = messageProcessorNodeIterator.next();
        visitorContext.annotate(new MessageSourceRegionContextAnnotation());
        annotate(child, visitorContext);
        visitorContext.deannotate(MessageSourceRegionContextAnnotation.class);
      }
      while (messageProcessorNodeIterator.hasNext()) {
        MessageProcessorNode child = messageProcessorNodeIterator.next();
        annotate(child, visitorContext);
      }
    } else {
      messageProcessorNode.getMessageProcessorNodes().forEach(child -> {
        annotate(child, visitorContext);
      });
    }
    visitorContext.logger().exit(messageProcessorNode.getName());
    return null;
  }

  @Override
  public Object visit(MuleApplicationNode muleApplicationNode, AnnotatingMuleAstVisitorContext visitorContext) {
    visitorContext.logger().enter(muleApplicationNode.getName());

    visitorContext.annotate(new MuleApplicationAnnotation(muleApplicationNode));
    getAnnotatorsRegistry().annotate(muleApplicationNode, visitorContext);
    muleApplicationNode.getMuleFlowNodes().forEach(muleFlowNode -> {
      annotate(muleFlowNode, visitorContext);
    });
    visitorContext.deannotate(MuleApplicationAnnotation.class);

    visitorContext.logger().exit(muleApplicationNode.getName());
    return null;
  }

  @Override
  public Object visit(MuleFlowNode muleFlowNode, AnnotatingMuleAstVisitorContext visitorContext) {
    visitorContext.logger().enter(muleFlowNode.getName());

    List<FunctionType> functionTypes = new ArrayList<>();

    visitorContext.getDataSenseProviderResolver().getApiKitMetadata().ifPresent(apiKitMetadata -> {
      apiKitMetadata.getMetadataForFlow(muleFlowNode.getName()).ifPresent(functionTypes::add);
    });

    visitorContext.getDataSenseProviderResolver().getSoapKitMetadata().ifPresent(soapKitMetadata -> {
      soapKitMetadata.getMetadataForFlow(muleFlowNode.getName()).ifPresent(functionTypes::add);
    });

    visitorContext.getDataSenseProviderResolver()
        .findMessageProcessorTypeDeclaration(muleFlowNode.getComponentModel())
        .ifPresent(messageProcessorTypeDeclaration -> {
          functionTypes.add(messageProcessorTypeDeclaration.getFunctionType());
          muleFlowNode.annotate(new MessageProcessorTypeDeclarationAnnotation(messageProcessorTypeDeclaration));
        });

    functionTypes.forEach(functionType -> {
      functionType.getParameters().stream().map(FunctionParameter::getType)
          .filter(metadataType -> metadataType instanceof MuleEventMetadataType)
          .map(metadataType -> TypeUtils.asEventType((MuleEventMetadataType) metadataType)).findFirst()
          .ifPresent(eventType -> {
            muleFlowNode.annotate(new UsesTypeAnnotation(eventType));
          });
      functionType.getReturnType().filter(metadataType -> metadataType instanceof MuleEventMetadataType)
          .map(metadataType -> TypeUtils.asEventType((MuleEventMetadataType) metadataType))
          .ifPresent(eventType -> {
            muleFlowNode.annotate(new DefinesTypeAnnotation(eventType));
          });
    });

    muleFlowNode.annotate(new ComponentLocationAnnotation(muleFlowNode.getComponentModel().getComponentLocation()));

    visitorContext.annotate(new MuleFlowAnnotation(muleFlowNode));
    getAnnotatorsRegistry().annotate(muleFlowNode, visitorContext);
    muleFlowNode.getRootMessageProcessorNode().ifPresent(messageProcessorNode -> annotate(messageProcessorNode, visitorContext));
    visitorContext.deannotate(MuleFlowAnnotation.class);

    visitorContext.logger().exit(muleFlowNode.getName());
    return null;
  }

  public void annotate(AstNode astNode, DataSenseProviderResolver dataSenseProviderResolver, AstNotification astNotification) {
    annotate(astNode, new AnnotatingMuleAstVisitorContext(astNotification, dataSenseProviderResolver));
  }

  public void annotate(AstNode astNode, AnnotatingMuleAstVisitorContext annotatingMuleAstVisitorContext) {
    astNode.accept(this, annotatingMuleAstVisitorContext);
  }

}
