package org.mule.datasense.impl.phases.typing.resolver;

import org.mule.datasense.impl.model.annotations.DefinesTypeAnnotation;
import org.mule.datasense.impl.model.annotations.MuleFlowAnnotation;
import org.mule.datasense.impl.model.annotations.UsesTypeAnnotation;
import org.mule.datasense.impl.model.ast.MessageProcessorNode;
import org.mule.datasense.impl.model.ast.MuleFlowNode;
import org.mule.datasense.impl.model.types.EventType;
import org.mule.datasense.impl.phases.builder.AstNodeBuilder;
import org.mule.datasense.impl.phases.builder.ComponentModelType;
import org.mule.datasense.impl.phases.builder.MessageProcessorNodeBuilder;
import org.mule.datasense.impl.phases.builder.MuleAstParseProvider;
import org.mule.datasense.impl.phases.typing.TypingMuleAstVisitor;
import org.mule.datasense.impl.phases.typing.TypingMuleAstVisitorContext;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.config.internal.model.ComponentModel;

import java.util.List;
import java.util.Optional;
import java.util.stream.Stream;

import static org.mule.datasense.impl.DefaultDataSense.COMPONENT_IDENTIFIER_FLOW_SCOPE_IN;
import static org.mule.datasense.impl.DefaultDataSense.COMPONENT_IDENTIFIER_FLOW_SCOPE_OUT;

public class FlowTypeResolver extends PipedChainTypeResolver {

  @Override
  public Optional<ComponentModelType> getComponentModelType() {
    return Optional.of(ComponentModelType.MESSAGE_PROCESSOR_NODE);
  }

  @Override
  public Optional<MuleAstParseProvider> getParseProvider() {
    return Optional.of(
                       (componentIdentifier, componentModel, componentModelType, messageProcessorNodeBuilders,
                        muleAstParserContext) -> {
                         return generateAst(componentIdentifier, componentModel, componentModelType,
                                            messageProcessorNodeBuilders);
                       });
  }

  private static Stream<MessageProcessorNode> getMessageProcessorNodeWithComponentIdentifier(MessageProcessorNode messageProcessorNode,
                                                                                             ComponentIdentifier componentIdentifier) {
    return messageProcessorNode.getMessageProcessorNodes()
        .filter(childMessageProcessorNode -> childMessageProcessorNode.getComponentIdentifier().equals(componentIdentifier));
  }

  public static Optional<MessageProcessorNode> getScopeInMessageProcessorNode(MessageProcessorNode messageProcessorNode) {
    return getMessageProcessorNodeWithComponentIdentifier(messageProcessorNode, COMPONENT_IDENTIFIER_FLOW_SCOPE_IN).findFirst();
  }

  public static Optional<MessageProcessorNode> getScopeOutMessageProcessorNode(MessageProcessorNode messageProcessorNode) {
    return getMessageProcessorNodeWithComponentIdentifier(messageProcessorNode, COMPONENT_IDENTIFIER_FLOW_SCOPE_OUT).findFirst();
  }

  private Optional<AstNodeBuilder> generateAst(ComponentIdentifier componentIdentifier, ComponentModel componentModel,
                                               ComponentModelType componentModelType,
                                               List<MessageProcessorNodeBuilder> messageProcessorNodeBuilders) {
    MessageProcessorNodeBuilder messageProcessorNodeBuilder = new MessageProcessorNodeBuilder(componentIdentifier);
    messageProcessorNodeBuilder.config(componentModel);
    messageProcessorNodeBuilder.componentModelType(componentModelType);
    messageProcessorNodeBuilder.messageProcessor(COMPONENT_IDENTIFIER_FLOW_SCOPE_IN, m -> {
      m.config(componentModel);
      m.synthetic();
    });
    messageProcessorNodeBuilders
        .forEach(messageProcessorNodeBuilder::messageProcessor);
    messageProcessorNodeBuilder.messageProcessor(COMPONENT_IDENTIFIER_FLOW_SCOPE_OUT, m -> {
      m.config(componentModel);
      m.synthetic();
    });

    return Optional.of(messageProcessorNodeBuilder);
  }

  public static class ScopeIn extends SingleNodeTypeResolver {

    @Override
    protected EventType resolve(MessageProcessorNode messageProcessorNode, EventType inputEventType,
                                TypingMuleAstVisitor typingMuleAstVisitor, TypingMuleAstVisitorContext visitorContext) {
      inputEventType =
          visitorContext.getAnnotation(MuleFlowAnnotation.class).map(MuleFlowAnnotation::getMuleFlowNode)
              .flatMap(muleFlowNode -> muleFlowNode.getAnnotation(UsesTypeAnnotation.class))
              .map(usesTypeAnnotation -> usesTypeAnnotation.getUsesEventType()).orElse(inputEventType);
      messageProcessorNode.annotate(new DefinesTypeAnnotation(inputEventType));

      return inputEventType;
    }

    @Override
    protected boolean isPropagates(MessageProcessorNode messageProcessorNode) {
      return true;
    }
  }


  public static class ScopeOut extends SingleNodeTypeResolver {

    @Override
    protected EventType resolve(MessageProcessorNode messageProcessorNode, EventType inputEventType,
                                TypingMuleAstVisitor typingMuleAstVisitor, TypingMuleAstVisitorContext visitorContext) {
      EventType eventType =
          visitorContext.getAnnotation(MuleFlowAnnotation.class).map(MuleFlowAnnotation::getMuleFlowNode)
              .flatMap(muleFlowNode -> muleFlowNode.getAnnotation(DefinesTypeAnnotation.class))
              .map(definesTypeAnnotation -> definesTypeAnnotation.getDefinesEventType()).orElse(new EventType());
      messageProcessorNode.annotate(new UsesTypeAnnotation(eventType));

      return inputEventType;
    }

    @Override
    protected boolean isPropagates(MessageProcessorNode messageProcessorNode) {
      return true;
    }
  }
}
