package org.mule.datasense.impl.phases.typing.resolver;

import com.google.common.base.Throwables;
import org.apache.commons.io.IOUtils;
import org.mule.datasense.impl.model.types.TypesHelper;
import org.mule.datasense.impl.phases.typing.resolver.errorhandling.ErrorHandlingUtils;
import org.mule.datasense.impl.util.LogSupport;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.model.MetadataType;

import java.io.IOException;

import static java.lang.String.format;
import static org.mule.datasense.impl.model.types.TypesHelper.getTypeBuilder;
import static org.mule.datasense.impl.model.types.TypesHelper.getTypeFromWeave;

public class GlobalBindingMetadataTypes implements LogSupport {

  public static final String GLOBAL_BINDING_MULE = "mule";
  public static final String GLOBAL_BINDING_SERVER = "server";
  public static final String GLOBAL_BINDING_APP = "app";
  public static final String GLOBAL_BINDING_FLOW = "flow";
  public static final String GLOBAL_BINDING_CORRELATION_ID = "correlationId";
  public static final String GLOBAL_BINDING_DATA_TYPE = "dataType";
  public static final String GLOBAL_BINDING_AUTHENTICATION = "authentication";
  public static final String GLOBAL_BINDING_ITEM_SEQUENCE_INFO = "itemSequenceInfo";
  public static final String GLOBAL_BINDING_ERROR = "error";

  private static final String GLOBAL_BINDING_DATA_TYPE_CLASS = "org.mule.runtime.api.metadata.DataType";
  private static final String GLOBAL_BINDING_AUTHENTICATION_CLASS = "org.mule.runtime.api.security.Authentication";
  private static final String GLOBAL_BINDING_ITEM_SEQUENCE_INFO_CLASS = "org.mule.runtime.api.message.ItemSequenceInfo";

  private static final String TYPEDVALUE_CLASS = "org.mule.runtime.api.metadata.TypedValue";

  public static MetadataType getTypeFromJavaClass(String clazz) {
    return TypesHelper.getTypeFromJavaClass(clazz, GlobalBindingMetadataTypes.class.getClassLoader())
        .orElseThrow(() -> new IllegalArgumentException(format("Failed to resolve class type %s", clazz)));
  }

  private static MetadataType metadataTypeFromWeaveResource(String resource, String type) {
    try {
      return getTypeFromWeave(IOUtils.toString(GlobalBindingMetadataTypes.class.getResourceAsStream(resource)), type)
          .orElseThrow(() -> new IllegalArgumentException(
                                                          format("Failed to resolve weave type %s", type)));
    } catch (IOException e) {
      throw Throwables.propagate(e);
    }
  }

  public static MetadataType correlationIdType() {
    final BaseTypeBuilder typeBuilder = getTypeBuilder();
    typeBuilder.stringType();
    return typeBuilder.build();
  }

  private static class DataTypeTypeHolder {

    private static final MetadataType singleton = getTypeFromJavaClass(GLOBAL_BINDING_DATA_TYPE_CLASS);

  }

  public static MetadataType dataTypeType() {
    return DataTypeTypeHolder.singleton;
  }

  private static class AuthenticationTypeHolder {

    private static final MetadataType singleton = getTypeFromJavaClass(GLOBAL_BINDING_AUTHENTICATION_CLASS);
  }

  public static MetadataType authenticationType() {
    return AuthenticationTypeHolder.singleton;
  }

  private static class ItemSequenceInfoTypeHolder {

    private static final MetadataType singleton = getTypeFromJavaClass(GLOBAL_BINDING_ITEM_SEQUENCE_INFO_CLASS);
  }

  public static MetadataType itemSequenceInfoType() {
    return ItemSequenceInfoTypeHolder.singleton;
  }

  private static class TypedValueTypeHolder {

    private static final MetadataType singleton = getTypeFromJavaClass(TYPEDVALUE_CLASS);
  }

  public static MetadataType typedValueType() {

    return TypedValueTypeHolder.singleton;
  }

  private static class AppTypeHolder {

    private static final MetadataType singleton = metadataTypeFromWeaveResource("app-type.dw", "DataWeaveArtifactContext");
  }

  private static class FlowTypeHolder {

    private static final MetadataType singleton = metadataTypeFromWeaveResource("flow-type.dw", "NamedObject");
  }

  public static MetadataType appType() {
    return AppTypeHolder.singleton;
  }

  public static MetadataType flowType() {
    return FlowTypeHolder.singleton;
  }

  private static class ServerTypeHolder {

    private static final MetadataType singleton = metadataTypeFromWeaveResource("server-type.dw", "ServerContext");
  }

  public static MetadataType serverType() {
    return ServerTypeHolder.singleton;
  }

  private static class MuleTypeHolder {

    private static final MetadataType singleton = metadataTypeFromWeaveResource("mule-type.dw", "MuleInstanceContext");
  }

  public static MetadataType muleType() {
    return MuleTypeHolder.singleton;
  }

  public static MetadataType errorType() {
    return ErrorHandlingUtils.errorType();
  }


}
