package org.mule.datasense.impl.util.bindings;

import org.apache.commons.lang.text.StrBuilder;
import org.mule.datasense.impl.model.types.TypeUtils;
import org.mule.datasense.impl.model.types.TypesHelper;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.message.api.MuleEventMetadataTypeBuilder;
import org.mule.metadata.message.api.el.ModuleDefinition;
import org.mule.metadata.message.api.el.ModuleIdentifier;
import org.mule.metadata.message.api.el.TypeBindings;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;

import static java.lang.String.format;

public class TypeBindingUtils {

  public static String toString(TypeBindings typeBindings) {
    if (typeBindings == null) {
      return "";
    }
    StrBuilder strBuilder = new StrBuilder();
    typeBindings.identifiers().forEach(identifier -> {
      strBuilder.append(format("%s: %s\n", identifier, TypesHelper.toString(typeBindings.lookup(identifier).orElseThrow(
                                                                                                                        IllegalArgumentException::new))));

    });
    typeBindings.modules().forEach(moduleDefinition -> {
      strBuilder.append(format("%s\n", moduleDefinition.getName()));
      moduleDefinition.identifiers().forEach(identifier -> {
        strBuilder
            .append(format("\t%s: %s\n", identifier, TypesHelper.toString(moduleDefinition.lookup(identifier).orElseThrow(
                                                                                                                          IllegalArgumentException::new))));

      });
    });
    return strBuilder.toString();
  }

  public static Map<String, MetadataType> toMap(TypeBindings typeBindings) {
    if (typeBindings == null) {
      return Collections.emptyMap();
    }

    Map<String, MetadataType> result = new LinkedHashMap<>();

    typeBindings.identifiers().forEach(identifier -> {
      typeBindings.lookup(identifier).ifPresent(metadataType -> {
        result.put(identifier, metadataType);
      });
    });
    typeBindings.modules().forEach(moduleDefinition -> {
      final ModuleIdentifier moduleIdentifier = moduleDefinition.getName();
      moduleDefinition.identifiers().forEach(identifier -> {
        moduleDefinition.lookup(identifier).ifPresent(metadataType -> {
          result.put(moduleIdentifier + ModuleIdentifier.NAME_SEPARATOR + identifier, metadataType);
        });
      });
    });

    return result;
  }

  public static TypeBindings fromMap(Map<String, MetadataType> typeBindingsMap) {
    if (typeBindingsMap == null) {
      return TypeBindings.builder().build();
    }

    Map<String, MetadataType> bindings = new LinkedHashMap<>();
    Map<String, Map<String, MetadataType>> modules = new LinkedHashMap<>();

    typeBindingsMap.forEach((qualifiedIdentifier, metadataType) -> {
      final String[] split = qualifiedIdentifier.split(ModuleIdentifier.NAME_SEPARATOR);
      if (split.length == 2) {
        String module = split[0];
        String identifier = split[1];
        Map<String, MetadataType> moduleBindings = modules.get(module);
        if (moduleBindings == null) {
          moduleBindings = new LinkedHashMap<>();
          modules.put(module, moduleBindings);
        }
        moduleBindings.put(identifier, metadataType);
      } else if (split.length == 1) {
        String identifier = split[0];
        bindings.put(identifier, metadataType);
      } else {
        throw new IllegalArgumentException(format("Illegal type binding identifier %s", qualifiedIdentifier));
      }
    });

    final TypeBindings.Builder builder = TypeBindings.builder();
    bindings.forEach(builder::addBinding);
    modules.forEach((module, moduleBindings) -> {
      final ModuleDefinition.Builder moduleBuilder = builder.module(module);
      moduleBindings.forEach(moduleBuilder::addElement);
    });
    return builder.build();
  }

  public static TypeBindings getTypeBindings(MetadataType muleEventMedataType, Map<String, MetadataType> globalBindings,
                                             Map<String, MetadataType> functionBindings) {
    return getTypeBindings(muleEventMedataType, fromMap(globalBindings), fromMap(functionBindings));
  }

  public static TypeBindings getTypeBindings(MetadataType muleEventMedataType, TypeBindings globalBindings,
                                             TypeBindings functionBindings) {
    final TypeBindings.Builder result = TypeBindings.builder();
    if (globalBindings != null) {
      result.addAll(globalBindings);
    }
    if (functionBindings != null) {
      result.addAll(functionBindings);
    }
    return TypeUtils.buildTypeBindings(TypeUtils.asMuleEventMetadataType(muleEventMedataType)
        .orElse(new MuleEventMetadataTypeBuilder().build()), result.build());
  }

}
