package org.mule.datasense.impl.phases.typing.resolver;

import static org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.DEFAULT_GROUP_NAME;

import org.mule.datasense.impl.model.annotations.DefinesTypeAnnotation;
import org.mule.datasense.impl.model.annotations.MuleApplicationAnnotation;
import org.mule.datasense.impl.model.annotations.ThrowsErrorsTypeAnnotation;
import org.mule.datasense.impl.model.annotations.UsesTypeAnnotation;
import org.mule.datasense.impl.model.ast.MessageProcessorNode;
import org.mule.datasense.impl.model.ast.MuleApplicationNode;
import org.mule.datasense.impl.model.ast.MuleFlowNode;
import org.mule.datasense.impl.model.reporting.NotificationMessages;
import org.mule.datasense.impl.model.types.EventType;
import org.mule.datasense.impl.model.types.TypeUtils;
import org.mule.datasense.impl.phases.builder.ComponentModelType;
import org.mule.datasense.impl.phases.typing.TypingMuleAstVisitor;
import org.mule.datasense.impl.phases.typing.TypingMuleAstVisitorContext;
import org.mule.runtime.api.i18n.I18nMessageFactory;
import org.mule.runtime.ast.api.ComponentParameterAst;

import java.util.Optional;

public class FlowRefTypeResolver extends SingleNodeTypeResolver {

  @Override
  public Optional<ComponentModelType> getComponentModelType() {
    return Optional.of(ComponentModelType.MESSAGE_PROCESSOR_NODE);
  }

  @Override
  protected EventType resolve(MessageProcessorNode messageProcessorNode, EventType inputEventType,
                              TypingMuleAstVisitor typingMuleAstVisitor,
                              TypingMuleAstVisitorContext visitorContext) {

    Optional<DefinesTypeAnnotation> definesTypeAnnotation = Optional.empty();

    ComponentParameterAst parameterAst = messageProcessorNode.getComponentModel().getParameter(DEFAULT_GROUP_NAME, "name");
    String referredFlowName = parameterAst.getResolvedRawValue();

    final MuleApplicationNode muleApplicationNode = visitorContext.getAnnotation(MuleApplicationAnnotation.class)
        .map(MuleApplicationAnnotation::getMuleApplicationNode)
        .orElseThrow(() -> new IllegalArgumentException("Mule application node not set."));
    final MuleFlowNode muleFlowNode = muleApplicationNode.findMuleFlowNode(referredFlowName).orElse(null);
    if (muleFlowNode != null) {
      messageProcessorNode
          .annotate(muleFlowNode.getAnnotation(UsesTypeAnnotation.class).orElse(new UsesTypeAnnotation(new EventType())));
      definesTypeAnnotation = muleFlowNode.getAnnotation(DefinesTypeAnnotation.class);
      if (!definesTypeAnnotation.isPresent()) {
        // attempt from inference
        typingMuleAstVisitor
            .resolveType(muleFlowNode, TypeUtils.createEventType(null), visitorContext.getAstNotification(),
                         muleApplicationNode);
        definesTypeAnnotation = muleFlowNode.getAnnotation(DefinesTypeAnnotation.class);
      }
      muleFlowNode
          .getAnnotation(ThrowsErrorsTypeAnnotation.class)
          .map(ThrowsErrorsTypeAnnotation::getErrors)
          .ifPresent(errorModels -> messageProcessorNode.annotate(new ThrowsErrorsTypeAnnotation(errorModels)));
    } else {
      visitorContext.getAstNotification().reportError(messageProcessorNode.getAstNodeLocation(), I18nMessageFactory
          .createStaticMessage(NotificationMessages.MSG_FAILED_TO_RESOLVE_FLOWREF, referredFlowName));
    }

    EventType definesEventType = definesTypeAnnotation.map(DefinesTypeAnnotation::getDefinesEventType).orElse(new EventType());
    definesEventType = processTarget(definesEventType, messageProcessorNode, visitorContext);
    messageProcessorNode.annotate(new DefinesTypeAnnotation(definesEventType));
    return definesEventType;
  }

  @Override
  protected boolean isPropagates(MessageProcessorNode messageProcessorNode) {
    return true;
  }
}
