package org.mule.datasense.impl;

import org.mule.datasense.api.DataSenseResolutionScope;
import org.mule.datasense.api.metadataprovider.ApplicationModel;
import org.mule.datasense.api.metadataprovider.DataSenseProvider;
import org.mule.datasense.impl.model.annotations.MuleFlowAnnotation;
import org.mule.datasense.impl.model.ast.AstNotification;
import org.mule.datasense.impl.model.ast.MessageProcessorNode;
import org.mule.datasense.impl.model.ast.MuleApplicationNode;
import org.mule.datasense.impl.model.types.EventType;
import org.mule.datasense.impl.model.types.TypeUtils;
import org.mule.datasense.impl.phases.annotators.AnnotatorsRegistry;
import org.mule.datasense.impl.phases.annotators.FunctionBindingsAnnotator;
import org.mule.datasense.impl.phases.annotators.GlobalBindingsAnnotator;
import org.mule.datasense.impl.phases.annotators.InfoAnnotator;
import org.mule.datasense.impl.phases.annotators.MUnitDeclarationAnnotator;
import org.mule.datasense.impl.phases.annotators.MessageProcessorTypeDeclarationAnnotator;
import org.mule.datasense.impl.phases.annotators.OperationCallAnnotator;
import org.mule.datasense.impl.phases.annotators.TransformAnnotator;
import org.mule.datasense.impl.phases.annotators.TypeResolverAnnotator;
import org.mule.datasense.impl.phases.annotators.UnknownTypeResolverAnnotator;
import org.mule.datasense.impl.phases.builder.ComponentModelType;
import org.mule.datasense.impl.phases.builder.MuleApplicationNodeBuilder;
import org.mule.datasense.impl.phases.builder.MuleAstParser;
import org.mule.datasense.impl.phases.builder.MuleAstParserContext;
import org.mule.datasense.impl.phases.typing.AnnotatingMuleAstVisitor;
import org.mule.datasense.impl.phases.typing.AnnotatingMuleAstVisitorContext;
import org.mule.datasense.impl.phases.typing.AstTyping;
import org.mule.datasense.impl.phases.typing.resolver.DataSenseTypeResolverRegistry;
import org.mule.datasense.impl.phases.typing.resolver.TypeResolver;
import org.mule.datasense.impl.phases.typing.resolver.TypeResolverRegistry;
import org.mule.datasense.impl.util.AstUtils;
import org.mule.datasense.impl.util.ComponentIdentifierUtils;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.TypedComponentIdentifier;
import org.mule.runtime.api.component.location.Location;
import org.mule.runtime.config.internal.model.ComponentModel;
import org.mule.runtime.core.api.processor.Processor;
import org.mule.runtime.core.api.transformer.Transformer;

import java.util.Optional;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DataSenseApplicationModel {

  private static final transient Logger logger = LoggerFactory.getLogger(DataSenseApplicationModel.class);
  private final DataSenseResolutionScope dataSenseResolutionScope;
  private final DataSenseProvider dataSenseProvider;
  private final AstNotification astNotification;
  private ApplicationModel applicationModel;
  private MuleApplicationNode muleApplicationNode;

  private DataSenseProviderResolver dataSenseProviderResolver;
  private TypeResolverRegistry typeResolverRegistry;


  public DataSenseApplicationModel(DataSenseResolutionScope dataSenseResolutionScope, ApplicationModel applicationModel,
                                   DataSenseProvider dataSenseProvider, AstNotification astNotification) {
    this(dataSenseResolutionScope, applicationModel, dataSenseProvider, new DataSenseTypeResolverRegistry(), astNotification);
  }

  public DataSenseApplicationModel(DataSenseResolutionScope dataSenseResolutionScope, ApplicationModel applicationModel,
                                   DataSenseProvider dataSenseProvider, TypeResolverRegistry typeResolverRegistry,
                                   AstNotification astNotification) {
    this.dataSenseResolutionScope = dataSenseResolutionScope;
    this.applicationModel = applicationModel;
    this.dataSenseProvider = dataSenseProvider;
    this.typeResolverRegistry = typeResolverRegistry;
    this.astNotification = astNotification;
  }

  public void build() {
    ApplicationModelResolver applicationModelResolver = new ApplicationModelResolver(applicationModel);
    dataSenseProviderResolver =
        new DataSenseProviderResolver(dataSenseProvider, applicationModelResolver, astNotification);

    ComponentModel rootComponentModel = applicationModel.findRootComponentModel();

    parse(rootComponentModel);
  }

  private void parse(ComponentModel rootComponentModel) {
    MuleAstParser muleAstParser = new MuleAstParser();

    MuleAstParserContext visitorContext = new MuleAstParserContext(applicationModel, dataSenseProvider, componentModel -> {
      ComponentIdentifier identifier = ComponentIdentifierUtils.createFromComponentModel(componentModel);

      ComponentModelType componentModelType =
          typeResolverRegistry.get(identifier).flatMap(TypeResolver::getComponentModelType)
              .orElse(componentModel.getComponentType().map(componentType -> {
                if (componentType.equals(TypedComponentIdentifier.ComponentType.SOURCE)) {
                  return ComponentModelType.MESSAGE_SOURCE_NODE;
                } else if (componentType.equals(TypedComponentIdentifier.ComponentType.OPERATION)) {
                  return ComponentModelType.MESSAGE_PROCESSOR_NODE;
                }
                return null;
              }).orElse(null));

      // TODO (gfernandes): remove this whenever batch has an extension model to resolve the componentType.
      if (componentModelType == null) {
        final Class<?> parsedComponentModelType = componentModel.getType();
        if (parsedComponentModelType != null) {
          if (Transformer.class.isAssignableFrom(parsedComponentModelType)
              || Processor.class.isAssignableFrom(parsedComponentModelType)) {
            componentModelType = ComponentModelType.MESSAGE_PROCESSOR_NODE;
          }
        }
      }

      return Optional.ofNullable(componentModelType);

    }, dataSenseResolutionScope, typeResolverRegistry);
    muleApplicationNode = muleAstParser.parse(rootComponentModel, visitorContext)
        .filter(astNodeBuilder -> astNodeBuilder instanceof MuleApplicationNodeBuilder)
        .map(astNodeBuilder -> ((MuleApplicationNodeBuilder) astNodeBuilder).build())
        .orElseThrow(() -> new RuntimeException("Failed to build application model."));
    AnnotatorsRegistry annotatorsRegistry = new AnnotatorsRegistry();
    annotatorsRegistry.add(new InfoAnnotator());
    AstTyping astTyping = new AstTyping(annotatorsRegistry, typeResolverRegistry);
    astTyping.annotate(muleApplicationNode, dataSenseProviderResolver, astNotification);
  }

  private AnnotatorsRegistry createAnnotatorsRegistry() {
    AnnotatorsRegistry annotatorsRegistry = new AnnotatorsRegistry();
    annotatorsRegistry.add(new GlobalBindingsAnnotator());
    annotatorsRegistry.add(new FunctionBindingsAnnotator());
    annotatorsRegistry.add(new MUnitDeclarationAnnotator());
    annotatorsRegistry.add(new TypeResolverAnnotator(typeResolverRegistry));
    //    annotatorsRegistry.add(new ConstructAnnotator());
    annotatorsRegistry.add(new OperationCallAnnotator());
    annotatorsRegistry.add(new UnknownTypeResolverAnnotator());
    annotatorsRegistry.add(new TransformAnnotator());
    annotatorsRegistry.add(new MessageProcessorTypeDeclarationAnnotator());
    return annotatorsRegistry;
  }

  public void resolve() {
    AnnotatorsRegistry annotatorsRegistry = createAnnotatorsRegistry();

    if (logger.isDebugEnabled()) {
      logger.debug(StringUtils.repeat("-", 10));
      AstUtils.dump(muleApplicationNode);
    }

    AstTyping astTyping = new AstTyping(annotatorsRegistry, typeResolverRegistry);

    if (logger.isDebugEnabled()) {
      logger.debug("== ANNOTATE " + StringUtils.repeat("-", 10));
    }

    astTyping.annotate(muleApplicationNode, dataSenseProviderResolver, astNotification);

    if (logger.isDebugEnabled()) {
      logger.debug("== RESOLVE " + StringUtils.repeat("-", 10));
    }

    EventType eventType = createInitialEvent();
    astTyping.resolveTypes(muleApplicationNode, eventType, astNotification,
                           dataSenseProviderResolver.getExpressionLanguageMetadataService());

    if (logger.isDebugEnabled()) {
      logger.debug(StringUtils.repeat("-", 10));
      AstUtils.dump(muleApplicationNode);
      logger.debug("== INCOMING " + StringUtils.repeat("-", 10));
    }

    astTyping.generateIncoming(muleApplicationNode, eventType, astNotification,
                               dataSenseProviderResolver.getExpressionLanguageMetadataService());

    if (logger.isDebugEnabled()) {
      logger.debug("== EXPECTED " + StringUtils.repeat("-", 10));
    }

    astTyping.generateExpected(muleApplicationNode, eventType, astNotification,
                               dataSenseProviderResolver.getExpressionLanguageMetadataService());

    if (logger.isDebugEnabled()) {
      logger.debug(StringUtils.repeat("-", 10));
      AstUtils.dump(muleApplicationNode);
    }
  }

  public boolean resolveComponent(Location location) {
    return findMessageProcessorNode(location).flatMap(messageProcessorNode -> {
      AnnotatorsRegistry annotatorsRegistry = createAnnotatorsRegistry();
      final String flowName = location.getGlobalName();
      return muleApplicationNode.findMuleFlowNode(flowName).map(muleFlowNode -> {
        final AnnotatingMuleAstVisitor annotatingMuleAstVisitor = new AnnotatingMuleAstVisitor(annotatorsRegistry);
        final AnnotatingMuleAstVisitorContext annotatingMuleAstVisitorContext =
            new AnnotatingMuleAstVisitorContext(astNotification, dataSenseProviderResolver);
        annotatingMuleAstVisitorContext.annotate(new MuleFlowAnnotation(muleFlowNode));
        annotatingMuleAstVisitor.annotate(messageProcessorNode,
                                          annotatingMuleAstVisitorContext);
        return true;
      });
    }).orElse(false);
  }


  private EventType createInitialEvent() {
    return TypeUtils.createEventType(null);
  }


  public Optional<ComponentModel> find(Location location) {
    return findMessageProcessorNode(location).map(MessageProcessorNode::getComponentModel);
  }

  public Optional<MessageProcessorNode> findMessageProcessorNode(Location location) {
    return muleApplicationNode.findMessageProcessorNode(location);
  }

  public MuleApplicationNode getMuleApplicationNode() {
    return muleApplicationNode;
  }

  public DataSenseProviderResolver getDataSenseProviderResolver() {
    return dataSenseProviderResolver;
  }
}
