package org.mule.datasense.impl.model.ast;

import org.mule.datasense.impl.model.annotations.ComponentLocationAnnotation;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.component.location.Location;
import org.mule.runtime.config.internal.model.ComponentModel;

import com.google.common.base.Preconditions;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Stream;

public class MuleFlowNode extends BaseAstNode {

  private final MessageProcessorNode rootMessageProcessorNode;
  private final ComponentModel componentModel;
  private AstNode parentAstNode;

  private Map<ComponentLocation, MessageProcessorNode> messageProcessorNodeMap;

  public MuleFlowNode(ComponentIdentifier componentIdentifier, ComponentModel componentModel,
                      MessageProcessorNode rootMessageProcessorNode) {
    super(componentIdentifier);
    this.componentModel = componentModel;
    Preconditions.checkNotNull(componentModel);
    setName(componentModel.getNameAttribute());
    this.rootMessageProcessorNode = rootMessageProcessorNode;
    messageProcessorNodeMap = new HashMap<>();
  }

  public MuleFlowNode(ComponentIdentifier componentIdentifier, ComponentModel componentModel) {
    this(componentIdentifier, componentModel, null);
  }

  public AstNode getParentAstNode() {
    return parentAstNode;
  }

  public ComponentModel getComponentModel() {
    return componentModel;
  }

  public void setParentAstNode(AstNode parentAstNode) {
    this.parentAstNode = parentAstNode;
  }

  public Optional<MessageProcessorNode> getRootMessageProcessorNode() {
    return Optional.ofNullable(rootMessageProcessorNode);
  }

  @Override
  public <T> Object accept(AstNodeVisitor<T> astNodeVisitor, T context) {
    return astNodeVisitor.visit(this, context);
  }

  public void register(ComponentLocation componentLocation, MessageProcessorNode messageProcessorNode) {
    Preconditions.checkState(!findMessageProcessorNode(componentLocation.getLocation()).isPresent(),
                             String.format("Node already registered for componentLocation %s", componentLocation.getLocation()));

    messageProcessorNodeMap.put(componentLocation, messageProcessorNode);
  }

  public Optional<MessageProcessorNode> findMessageProcessorNode(String location) {
    return messageProcessorNodeMap.entrySet().stream()
        .filter(messageProcessorNode -> messageProcessorNode.getKey().getLocation().equals(location)).map(
                                                                                                          Map.Entry::getValue)
        .findFirst();
  }

  public Stream<MessageProcessorNode> findMessageProcessorNodes(boolean includeRoot) {
    return messageProcessorNodeMap.values().stream()
        .filter(messageProcessorNode -> includeRoot || !messageProcessorNode.isRootMessageProcessorNode());
  }

  public Stream<MessageProcessorNode> findMessageProcessorNodes() {
    return findMessageProcessorNodes(false);
  }

}
