package org.mule.datasense.impl.phases.annotators;

import org.mule.datasense.impl.model.ast.MessageProcessorNode;
import org.mule.datasense.impl.model.operation.InputMapping;
import org.mule.datasense.impl.model.operation.OperationCall;
import org.mule.datasense.impl.model.operation.OperationCallBuilder;
import org.mule.datasense.impl.model.types.TypesHelper;
import org.mule.datasense.impl.util.ExpressionLanguageUtils;
import org.mule.datasense.impl.util.extension.DslElementModelVisitor;
import org.mule.datasense.impl.util.extension.DslElementModelWalker;
import org.mule.datasense.impl.util.extension.ExtensionUtils;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectFieldType;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.config.internal.model.ComponentModel;
import org.mule.runtime.config.api.dsl.model.DslElementModel;
import org.mule.runtime.extension.api.property.MetadataKeyPartModelProperty;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Optional;

public abstract class StaticDslBaseAnnotator extends BaseOperationCallBuilderAnnotator {

  protected static class OperationCallDslElementModelVisitor implements DslElementModelVisitor {

    private final OperationCallBuilder operationCallBuilder;
    private boolean hasDynamicType;
    private List<ParameterModel> incompleteForDynamicMetadata;

    public OperationCallDslElementModelVisitor(OperationCallBuilder operationCallBuilder) {
      this.operationCallBuilder = operationCallBuilder;
      hasDynamicType = false;
      incompleteForDynamicMetadata = new ArrayList<>();
    }

    @Override
    public void visitOperationModel(DslElementModel<OperationModel> operationModelDslElementModel,
                                    DslElementModelWalker dslElementModelWalker) {
      final OperationModel model = operationModelDslElementModel.getModel();
      operationCallBuilder.name(model.getName());
      operationModelDslElementModel.getContainedElements().forEach(dslElementModel -> {
        dslElementModelWalker.walkDslElementModel(dslElementModel, this);
      });
      operationCallBuilder.errors(model.getErrorModels());
      operationCallBuilder.returnType(TypesHelper.getMessageMetadataTypeBuilder()
          .payload(model.getOutput().getType())
          .attributes(model.getOutputAttributes().getType()).build());
      hasDynamicType |= model.getOutput().hasDynamicType() || model.getOutputAttributes().hasDynamicType();
      if (hasDynamicType) {
        incompleteForDynamicMetadata = incompleteForDynamicMetadata(operationCallBuilder, model);
      }
    }

    private static List<ParameterModel> incompleteForDynamicMetadata(OperationCallBuilder operationCallBuilder,
                                                                     ParameterizedModel parameterizedModel) {
      List<ParameterModel> incompleteParameters = new ArrayList<>();

      if (operationCallBuilder != null) {
        final OperationCall operationCall = operationCallBuilder.build();
        final List<ParameterModel> allParameterModels = parameterizedModel.getAllParameterModels();
        if (allParameterModels != null) {
          for (ParameterModel parameterModel : allParameterModels) {
            final MetadataKeyPartModelProperty metadataKeyPartModelProperty =
                parameterModel.getModelProperty(MetadataKeyPartModelProperty.class).orElse(null);
            if (metadataKeyPartModelProperty != null) {
              if (!operationCall.getInputMapping(parameterModel.getName()).isPresent()) {
                incompleteParameters.add(parameterModel);
              } ;
            }
          }
        }
      }

      return incompleteParameters;
    }

    @Override
    public void visitSourceModel(DslElementModel<SourceModel> sourceModelDslElementModel,
                                 DslElementModelWalker dslElementModelWalker) {
      final SourceModel model = sourceModelDslElementModel.getModel();
      operationCallBuilder.name(model.getName());
      sourceModelDslElementModel.getContainedElements().forEach(dslElementModel -> {
        dslElementModelWalker.walkDslElementModel(dslElementModel, this);
      });
      operationCallBuilder.errors(model.getErrorModels());
      operationCallBuilder.returnType(TypesHelper.getMessageMetadataTypeBuilder()
          .payload(model.getOutput().getType())
          .attributes(model.getOutputAttributes().getType()).build());
      hasDynamicType |= model.getOutput().hasDynamicType() || model.getOutputAttributes().hasDynamicType();
      incompleteForDynamicMetadata = incompleteForDynamicMetadata(operationCallBuilder, model);
    }

    private void addParameter(String name, MetadataType metadataType, String value) {
      operationCallBuilder.input(inputMappingBuilder -> {
        inputMappingBuilder
            .parameter(inputParameterBuilder -> {
              inputParameterBuilder
                  .name(name)
                  .type(metadataType);
            })
            .argument(inputArgumentBuilder -> {
              ExpressionLanguageUtils.extractExpression(value).ifPresent(inputArgumentBuilder::expression);
              inputArgumentBuilder.value(value);

            });
      });
    }

    @Override
    public void visitParameterGroupModel(DslElementModel<ParameterGroupModel> parameterGroupModelDslElementModel,
                                         DslElementModelWalker dslElementModelWalker) {
      parameterGroupModelDslElementModel.getContainedElements().forEach(dslElementModel -> {
        dslElementModelWalker.walkDslElementModel(dslElementModel, this);
      });
    }

    @Override
    public void visitParameterModel(DslElementModel<ParameterModel> parameterModelDslElementModel,
                                    DslElementModelWalker dslElementModelWalker) {
      final ParameterModel parameterModel = parameterModelDslElementModel.getModel();
      hasDynamicType |= parameterModel.hasDynamicType();

      final Optional<String> valueOptional = parameterModelDslElementModel.getValue();
      if (valueOptional.isPresent()) {
        addParameter(parameterModel.getName(), parameterModel.getType(), valueOptional.get());
      } else {
        parameterModelDslElementModel.getContainedElements().forEach(dslElementModel -> {
          dslElementModelWalker.walkDslElementModel(dslElementModel, this);
        });
      }
    }

    @Override
    public void visitMetadataType(DslElementModel<MetadataType> metadataTypeDslElementModel,
                                  DslElementModelWalker dslElementModelWalker) {
      final MetadataType model = metadataTypeDslElementModel.getModel();
      final Optional<String> valueOptional = metadataTypeDslElementModel.getValue();
      if (valueOptional.isPresent()) {
        MetadataType type;
        if (model instanceof ObjectFieldType) {
          type = ((ObjectFieldType) model).getValue();
        } else {
          type = model;
        }
        addParameter(metadataTypeDslElementModel.getIdentifier().map(componentIdentifier -> componentIdentifier.getName())
            .orElse(""), type, valueOptional.get());
      } else {
        metadataTypeDslElementModel.getContainedElements().forEach(dslElementModel -> {
          dslElementModelWalker.walkDslElementModel(dslElementModel, this);
        });
      }
    }

    public boolean isHasDynamicType() {
      return hasDynamicType;
    }

    public Collection<ParameterModel> isIncompleteForDynamicMetadata() {
      return incompleteForDynamicMetadata;
    }
  }

  //  protected Optional<String> findTarget(MessageProcessorNode messageProcessorNode) {
  //    return ExtensionUtils.findTarget(messageProcessorNode.getComponentModel());
  //  }
  //
  //  protected Optional<String> findTarget(DslElementModel<ParameterizedModel> dslElementModel) {
  //    return ExtensionUtils.findParameter(dslElementModel.getModel(), ExtensionUtils.PARAMETER_TARGET).map(parameterModel -> {
  //      return ExtensionUtils.findTarget(dslElementModel.getConfiguration());
  //    });
  //  }
}
