package org.mule.datasense.impl.phases.typing.resolver;

import static java.util.Optional.ofNullable;
import static org.mule.datasense.impl.model.types.TypesHelper.getTypeBuilder;
import static org.mule.datasense.impl.phases.builder.MuleAstParser.MULE_AGGREGATORS;
import org.mule.datasense.impl.model.annotations.InnerUsesTypeAnnotation;
import org.mule.datasense.impl.model.ast.MessageProcessorNode;
import org.mule.datasense.impl.model.reporting.NotificationMessages;
import org.mule.datasense.impl.model.types.EventType;
import org.mule.datasense.impl.model.types.TypeUtils;
import org.mule.datasense.impl.model.types.TypesHelper;
import org.mule.datasense.impl.phases.builder.ComponentModelType;
import org.mule.datasense.impl.phases.typing.TypingMuleAstVisitor;
import org.mule.datasense.impl.phases.typing.TypingMuleAstVisitorContext;
import org.mule.datasense.impl.util.ComponentIdentifierUtils;
import org.mule.datasense.impl.util.ComponentModelUtils;
import org.mule.datasense.impl.util.ExpressionLanguageUtils;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.ObjectTypeBuilder;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.message.api.MessageMetadataTypeBuilder;
import org.mule.metadata.message.api.MuleEventMetadataType;
import org.mule.metadata.message.api.MuleEventMetadataTypeBuilder;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.metadata.ExpressionLanguageMetadataService;
import org.mule.runtime.config.internal.model.ComponentModel;

import java.util.Optional;

public class AggregatorTypeResolver extends StructuralNodeTypeResolver {

  private static final String PARAMETER_CONTENT = "content";
  private static final String PARAMETER_CONTENT_DEFAULT = "#[message]";
  private static final String AGGREGATOR_ATTRIBUTES_CLASS = "org.mule.extension.aggregator.internal.routes.AggregatorAttributes";
  private static final ComponentIdentifier PARAMETER_CONTENT_IDENTIFIER =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_AGGREGATORS, PARAMETER_CONTENT);

  @Override
  public Optional<ComponentModelType> getComponentModelType() {
    return Optional.of(ComponentModelType.MESSAGE_PROCESSOR_NODE);
  }

  private Optional<MetadataType> resolveInnerElementType(MessageProcessorNode messageProcessorNode, EventType inputEventType,
                                                         TypingMuleAstVisitorContext visitorContext) {
    return ofNullable(messageProcessorNode.getComponentModel())
        .map(componentModel -> {
          final String contentData =
              Optional.ofNullable(getContentParameterValue(messageProcessorNode)).orElse(PARAMETER_CONTENT_DEFAULT);

          return ExpressionLanguageUtils.extractExpression(contentData)
              .map(e -> ExpressionLanguageUtils
                  .resolveExpressionType(e, inputEventType,
                                         visitorContext.getTypeBindings(),
                                         visitorContext.getExpressionLanguageMetadataService(),
                                         new ExpressionLanguageMetadataService.MessageCallback() {

                                           @Override
                                           public void warning(String message,
                                                               ExpressionLanguageMetadataService.MessageLocation messageLocation) {
                                             visitorContext.getAstNotification()
                                                 .reportWarning(messageProcessorNode.getAstNodeLocation(),
                                                                NotificationMessages
                                                                    .MSG_SCRIPTING_LANGUAGE_WARNING(e, message));
                                           }

                                           @Override
                                           public void error(String message,
                                                             ExpressionLanguageMetadataService.MessageLocation messageLocation) {
                                             visitorContext.getAstNotification()
                                                 .reportError(messageProcessorNode.getAstNodeLocation(),
                                                              NotificationMessages.MSG_SCRIPTING_LANGUAGE_ERROR(e, message));
                                           }

                                         }))
              .orElse(null);
        });
  }

  private String getContentParameterValue(MessageProcessorNode messageProcessorNode) {
    if (messageProcessorNode == null) {
      return null;
    }

    String parameterContent = messageProcessorNode.getComponentModel().getParameters().get(PARAMETER_CONTENT);
    if (parameterContent == null) {
      parameterContent = ComponentModelUtils.collectComponentModelById(messageProcessorNode, PARAMETER_CONTENT_IDENTIFIER)
          .map(ComponentModel::getTextContent).orElse(null);
    }
    return parameterContent;
  }

  public static MetadataType aggregatorAttributesType() {
    return TypesHelper.getTypeFromJavaClass(AGGREGATOR_ATTRIBUTES_CLASS, AggregatorTypeResolver.class.getClassLoader())
        .orElse(fallbackAggregatorAttributesType());
  }

  private static MetadataType fallbackAggregatorAttributesType() {
    final BaseTypeBuilder aggregatorAttributesTypeBuilder = getTypeBuilder();

    final ObjectTypeBuilder typeBuilder = aggregatorAttributesTypeBuilder.objectType();
    typeBuilder.addField().key("groupId").value().stringType();
    typeBuilder.addField().key("firstValueArrivalTime").value().numberType();
    typeBuilder.addField().key("lastValueArrivalTime").value().numberType();
    typeBuilder.addField().key("groupComplete").value().booleanType();

    return aggregatorAttributesTypeBuilder.build();
  }

  private EventType buildInnerEventType(MessageProcessorNode messageProcessorNode, EventType inputEventType,
                                        TypingMuleAstVisitorContext visitorContext) {

    final MuleEventMetadataTypeBuilder muleEventMetadataTypeBuilder = TypesHelper.getMuleEventMetadataTypeBuilder();
    final MessageMetadataTypeBuilder message = muleEventMetadataTypeBuilder.message();
    message.payload().arrayType().of(resolveInnerElementType(messageProcessorNode, inputEventType, visitorContext)
        .orElse(TypesHelper.getTypeBuilder().anyType().build()));
    message.attributes(aggregatorAttributesType());

    final MuleEventMetadataType muleEventMetadataType = muleEventMetadataTypeBuilder.build();
    return TypeUtils.asEventType(muleEventMetadataType);
  }


  @Override
  protected EventType resolve(MessageProcessorNode messageProcessorNode, EventType inputEventType,
                              TypingMuleAstVisitor typingMuleAstVisitor,
                              TypingMuleAstVisitorContext typingMuleAstVisitorContext) {

    final EventType innerEventType = buildInnerEventType(messageProcessorNode, inputEventType, typingMuleAstVisitorContext);
    messageProcessorNode.annotate(new InnerUsesTypeAnnotation(innerEventType));

    messageProcessorNode.getMessageProcessorNodes().forEach(pipedMessageProcessorNode -> {
      typingMuleAstVisitor.resolveType(pipedMessageProcessorNode, TypeUtils.override(inputEventType, innerEventType),
                                       typingMuleAstVisitorContext);
    });

    return new EventType();
  }

  @Override
  protected boolean isPropagates(MessageProcessorNode messageProcessorNode) {
    return true;
  }

  @Override
  protected boolean isSequential() {
    return false;
  }

}
