package org.mule.datasense.impl.phases.typing.resolver;

import static java.lang.String.format;
import static java.util.Collections.emptySet;
import static org.mule.datasense.impl.model.types.TypesHelper.getTypeBuilder;
import org.mule.datasense.impl.model.types.TypesHelper;
import org.mule.datasense.impl.phases.typing.resolver.errorhandling.ErrorHandlingUtils;
import org.mule.datasense.impl.util.LogSupport;
import org.mule.metadata.MetadataFormats;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;

import com.google.common.base.Throwables;

import java.io.IOException;
import java.util.Optional;

import org.apache.commons.io.IOUtils;
import org.mule.runtime.api.metadata.ExpressionLanguageMetadataService;

public class GlobalBindingMetadataTypes implements LogSupport {

  public static final String GLOBAL_BINDING_MULE = "mule";
  public static final String GLOBAL_BINDING_SERVER = "server";
  public static final String GLOBAL_BINDING_APP = "app";
  public static final String GLOBAL_BINDING_FLOW = "flow";
  public static final String GLOBAL_BINDING_CORRELATION_ID = "correlationId";
  public static final String GLOBAL_BINDING_DATA_TYPE = "dataType";
  public static final String GLOBAL_BINDING_AUTHENTICATION = "authentication";
  public static final String GLOBAL_BINDING_ITEM_SEQUENCE_INFO = "itemSequenceInfo";
  public static final String GLOBAL_BINDING_ERROR = "error";

  private static final String GLOBAL_BINDING_DATA_TYPE_CLASS = "org.mule.runtime.api.metadata.DataType";
  private static final String GLOBAL_BINDING_AUTHENTICATION_CLASS = "org.mule.runtime.api.security.Authentication";
  private static final String GLOBAL_BINDING_ITEM_SEQUENCE_INFO_CLASS = "org.mule.runtime.api.message.ItemSequenceInfo";
  private static final String ERROR_CLASS = "org.mule.runtime.api.message.Error";
  private static final String TYPEDVALUE_CLASS = "org.mule.runtime.api.metadata.TypedValue";

  public static MetadataType getTypeFromJavaClass(String clazz) {
    return TypesHelper.getTypeFromJavaClass(clazz, GlobalBindingMetadataTypes.class.getClassLoader())
        .orElseThrow(() -> new IllegalArgumentException(format("Failed to resolve class type %s", clazz)));
  }

  private static Optional<MetadataType> getTypeFromWeave(String weaveTypeDef, String typeId, MetadataFormat metadataFormat,
                                                         ExpressionLanguageMetadataService expressionLanguageMetadataService) {
    return expressionLanguageMetadataService
        .createTypeLoader(weaveTypeDef, Optional.ofNullable(metadataFormat).orElse(MetadataFormats.JAVA)).load(typeId);
  }

  private static Optional<MetadataType> getTypeFromWeave(String weaveTypeDef, String typeId,
                                                         ExpressionLanguageMetadataService expressionLanguageMetadataService) {
    return getTypeFromWeave(weaveTypeDef, typeId, null, expressionLanguageMetadataService);
  }


  private static MetadataType metadataTypeFromWeaveResource(String resource, String type,
                                                            ExpressionLanguageMetadataService expressionLanguageMetadataService) {
    try {
      return getTypeFromWeave(IOUtils.toString(GlobalBindingMetadataTypes.class.getResourceAsStream(resource)), type,
                              expressionLanguageMetadataService)
                                  .orElseThrow(() -> new IllegalArgumentException(
                                                                                  format("Failed to resolve weave type %s",
                                                                                         type)));
    } catch (IOException e) {
      throw Throwables.propagate(e);
    }
  }


  private final ExpressionLanguageMetadataService expressionLanguageMetadataService;

  private final MetadataType correlationIdType;
  private final MetadataType dataTypeType;
  private final MetadataType authenticationType;
  private final MetadataType itemSequenceInfoType;
  private final MetadataType typedValueType;
  private final MetadataType appType;
  private final MetadataType serverType;
  private final MetadataType muleType;
  private final MetadataType errorType;
  private final MetadataType flowType;

  public GlobalBindingMetadataTypes(
                                    ExpressionLanguageMetadataService expressionLanguageMetadataService) {
    this.expressionLanguageMetadataService = expressionLanguageMetadataService;

    correlationIdType = createCorrelationIdType();
    dataTypeType = createDataTypeType();
    authenticationType = createAuthenticationType();
    itemSequenceInfoType = createItemSequenceInfoType();
    typedValueType = createTypedValueType();
    appType = createAppType();
    serverType = createServerType();
    muleType = createMuleType();
    errorType = createErrorType();
    flowType = createFlowType();
  }

  private MetadataType createCorrelationIdType() {
    final BaseTypeBuilder typeBuilder = getTypeBuilder();
    typeBuilder.stringType();
    return typeBuilder.build();
  }

  public MetadataType correlationIdType() {
    return correlationIdType;
  }

  private MetadataType createDataTypeType() {
    return getTypeFromJavaClass(GLOBAL_BINDING_DATA_TYPE_CLASS);
  }

  public MetadataType dataTypeType() {
    return dataTypeType;
  }

  private MetadataType createAuthenticationType() {
    return getTypeFromJavaClass(GLOBAL_BINDING_AUTHENTICATION_CLASS);
  }

  public MetadataType authenticationType() {
    return authenticationType;
  }

  private MetadataType createItemSequenceInfoType() {
    return getTypeFromJavaClass(GLOBAL_BINDING_ITEM_SEQUENCE_INFO_CLASS);
  }

  public MetadataType itemSequenceInfoType() {
    return itemSequenceInfoType;
  }

  private MetadataType createTypedValueType() {
    return getTypeFromJavaClass(TYPEDVALUE_CLASS);
  }

  public MetadataType typedValueType() {
    return typedValueType;
  }

  private MetadataType createAppType() {
    return metadataTypeFromWeaveResource("app-type.dw", "DataWeaveArtifactContext", expressionLanguageMetadataService);
  }

  public MetadataType appType() {
    return appType;
  }

  private MetadataType createServerType() {
    return metadataTypeFromWeaveResource("server-type.dw", "ServerContext", expressionLanguageMetadataService);
  }

  public MetadataType serverType() {
    return serverType;
  }

  private MetadataType createMuleType() {
    return metadataTypeFromWeaveResource("mule-type.dw", "MuleInstanceContext", expressionLanguageMetadataService);
  }

  public MetadataType muleType() {
    return muleType;
  }

  private MetadataType createErrorType() {
    return TypesHelper.getTypeFromJavaClass(ERROR_CLASS, GlobalBindingMetadataTypes.class.getClassLoader())
        .orElse(ErrorHandlingUtils.errorType(emptySet()));
  }

  public MetadataType errorType() {
    return errorType;
  }

  private MetadataType createFlowType() {
    return metadataTypeFromWeaveResource("flow-type.dw", "NamedObject", expressionLanguageMetadataService);
  }

  public MetadataType flowType() {
    return flowType;
  }

}
