package org.mule.datasense.api.metadataprovider;

import static java.lang.String.format;

import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.api.metadata.ExpressionLanguageMetadataService;
import org.mule.runtime.api.metadata.MetadataKeysContainer;
import org.mule.runtime.api.metadata.resolving.MetadataResult;
import org.mule.runtime.apikit.metadata.api.MetadataBuilder;

import java.util.Optional;
import java.util.Set;

import org.mule.runtime.config.api.dsl.model.DslElementModelFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CachedDataSenseProvider implements DataSenseProvider {

  static final transient Logger logger = LoggerFactory.getLogger(CachedDataSenseProvider.class);
  private final DataSenseMetadataCacheProvider dataSenseMetadataCacheProvider;
  private final DataSenseProvider delegate;

  public CachedDataSenseProvider(DataSenseMetadataCacheProvider dataSenseMetadataCacheProvider,
                                 DataSenseProvider delegate) {
    this.delegate = delegate;
    this.dataSenseMetadataCacheProvider = dataSenseMetadataCacheProvider;
  }

  @Override
  public Set<ExtensionModel> getExtensions() {
    return delegate.getExtensions();
  }

  @Override
  public Optional<DataSenseMetadataProvider> getDataSenseMetadataProvider() {
    return delegate
        .getDataSenseMetadataProvider()
        .map(dataSenseMetadataProvider -> new CachedDataSenseMetadataProvider(dataSenseMetadataProvider,
                                                                              dataSenseMetadataCacheProvider));
  }

  @Override
  public Optional<DataSenseConfiguration> getDataSenseConfiguration() {
    return delegate.getDataSenseConfiguration();
  }

  @Override
  public ExpressionLanguageMetadataService getExpressionLanguageMetadataService() {
    return delegate.getExpressionLanguageMetadataService();
  }

  @Override
  public ExpressionLanguage getExpressionLanguage() {
    return delegate.getExpressionLanguage();
  }

  @Override
  public MetadataBuilder getApikitMetadataBuilder() {
    return delegate.getApikitMetadataBuilder();
  }

  @Override
  public DslElementModelFactory getDslElementModelFactory() {
    return delegate.getDslElementModelFactory();
  }

  private class CachedDataSenseMetadataProvider implements DataSenseMetadataProvider {

    private final DataSenseMetadataProvider delegate;
    private final DataSenseMetadataCacheProvider dataSenseMetadataCacheProvider;

    public CachedDataSenseMetadataProvider(DataSenseMetadataProvider delegate,
                                           DataSenseMetadataCacheProvider dataSenseMetadataCacheProvider) {
      this.delegate = delegate;
      this.dataSenseMetadataCacheProvider = dataSenseMetadataCacheProvider;
    }

    public DataSenseMetadataProvider getDelegate() {
      return delegate;
    }

    @Override
    public MetadataResult<OperationModel> getOperationMetadata(CompatibleComponentAst componentModelAst) {
      try {
        return dataSenseMetadataCacheProvider.getOperationMetadata(componentModelAst, () -> Optional
            .ofNullable(getDelegate()
                .getOperationMetadata(componentModelAst))
            .orElseThrow(() -> new RuntimeException(format("Failed to resolve operation metadata for component path %s.",
                                                           componentModelAst.getLocation()))));
      } catch (Exception e) {
        logger.error(format("Failed to resolve operation metadata for component path %s.",
                            componentModelAst.getLocation()),
                     e);
        return null;
      }
    }

    @Override
    public MetadataResult<SourceModel> getSourceMetadata(CompatibleComponentAst componentModelAst) {
      try {
        return dataSenseMetadataCacheProvider
            .getSourceMetadata(componentModelAst, () -> Optional
                .ofNullable(getDelegate().getSourceMetadata(componentModelAst))
                .orElseThrow(() -> new RuntimeException(format("Failed to resolve source metadata for component path %s.",
                                                               componentModelAst.getLocation()))));
      } catch (Exception e) {
        logger.error(format("Failed to resolve source metadata for component path %s.",
                            componentModelAst.getLocation()),
                     e);
        return null;
      }
    }

    @Override
    public MetadataResult<MetadataKeysContainer> getMetadataKeys(CompatibleComponentAst componentModelAst) {
      try {
        return dataSenseMetadataCacheProvider
            .getMetadataKeys(componentModelAst,
                             () -> Optional.ofNullable(getDelegate().getMetadataKeys(componentModelAst))
                                 .orElseThrow(() -> new RuntimeException(format("Failed to resolve metadata keys for component path %s.",
                                                                                componentModelAst.getLocation()))));
      } catch (Exception e) {
        logger.error(format("Failed to resolve metadata keys for component path %s.",
                            componentModelAst.getLocation()),
                     e);
        return null;
      }
    }
  }
}
