package org.mule.datasense.impl.phases.typing.resolver;

import static org.mule.datasense.impl.DefaultDataSense.COMPONENT_IDENTIFIER_FOREACH_SCOPE_IN;
import static org.mule.datasense.impl.model.types.TypeUtils.removeNullsFromUnionMetadataType;
import static org.mule.datasense.impl.phases.builder.MuleAstParser.MULE_CORE;

import org.mule.datasense.impl.model.annotations.DefinesTypeAnnotation;
import org.mule.datasense.impl.model.annotations.UsesTypeAnnotation;
import org.mule.datasense.impl.model.ast.AstNotification;
import org.mule.datasense.impl.model.ast.MessageProcessorNode;
import org.mule.datasense.impl.model.reporting.NotificationMessages;
import org.mule.datasense.impl.model.types.EventType;
import org.mule.datasense.impl.model.types.TypeUtils;
import org.mule.datasense.impl.model.types.TypesHelper;
import org.mule.datasense.impl.phases.builder.AstNodeBuilder;
import org.mule.datasense.impl.phases.builder.ComponentModelType;
import org.mule.datasense.impl.phases.builder.MessageProcessorNodeBuilder;
import org.mule.datasense.impl.phases.builder.MuleAstParseProvider;
import org.mule.datasense.impl.phases.scoping.ExpectedAstVisitor;
import org.mule.datasense.impl.phases.scoping.ExpectedAstVisitorContext;
import org.mule.datasense.impl.phases.typing.TypingMuleAstVisitor;
import org.mule.datasense.impl.phases.typing.TypingMuleAstVisitorContext;
import org.mule.datasense.impl.util.ComponentIdentifierUtils;
import org.mule.datasense.impl.util.ComponentModelUtils;
import org.mule.datasense.impl.util.ExpressionLanguageUtils;
import org.mule.metadata.api.builder.UnionTypeBuilder;
import org.mule.metadata.api.model.ArrayType;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.message.api.MessageMetadataType;
import org.mule.metadata.message.api.MuleEventMetadataType;
import org.mule.metadata.message.api.MuleEventMetadataTypeBuilder;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.metadata.ExpressionLanguageMetadataService;
import org.mule.runtime.config.internal.model.ComponentModel;

import java.util.LinkedHashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

public class ForEachTypeResolver extends PipedChainTypeResolver {

  private static final String ATTR_BATCH_SIZE = "batchSize";
  private static final String ATTR_COLLECTION = "collection";
  private static final String PARAMETER_COLLECTION = "collection";
  private static final String ATTR_COUNTER_VARIABLE_NAME = "counterVariableName";
  private static final String ATTR_MESSAGE_VARIABLE_NAME = "rootMessageVariableName";

  private static final String DEFAULT_COUNTER_VARIABLE_NAME = "counter";
  private static final String DEFAULT_MESSAGE_VARIABLE_NAME = "rootMessage";
  private static final String DEFAULT_COLLECTION_EXPRESSION = "#[payload]";
  private static final int DEFAULT_BATCH_SIZE = 1;
  private static final ComponentIdentifier PARAMETER_COLLECTION_IDENTIFIER =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE, PARAMETER_COLLECTION);

  private static boolean isBatchMode(ComponentModel componentModel) {
    boolean result;
    final String expression = componentModel.getParameters().get(ATTR_BATCH_SIZE);
    if (expression == null) {
      // batchSize 1
      result = false;
    } else {
      if (ExpressionLanguageUtils.isExpression(expression)) {
        result = true;
      } else {
        try {
          result = Integer.parseInt(expression) > 1;
        } catch (NumberFormatException e) {
          result = false;
        }
      }
    }
    return result;
  }

  public static Optional<MetadataType> buildInnerPayloadType(MetadataType metadataType) {
    metadataType = removeNullsFromUnionMetadataType(metadataType);

    if (metadataType instanceof ArrayType) {
      final ArrayType arrayType = (ArrayType) metadataType;
      return Optional.of(arrayType.getType());
    } else if (metadataType instanceof ObjectType) {
      final ObjectType objectType = (ObjectType) metadataType;
      Set<MetadataType> metadataTypes = new LinkedHashSet<>();
      objectType.getFields().forEach(objectFieldType -> {
        metadataTypes.add(objectFieldType.getValue());
      });

      if (metadataTypes.size() == 0) {
        return Optional.empty();
      } else if (metadataTypes.size() == 1) {
        return Optional.of(metadataTypes.iterator().next());
      } else {
        final UnionTypeBuilder unionTypeBuilder = TypesHelper.getTypeBuilder().unionType();
        metadataTypes.forEach(unionTypeBuilder::of);
        return Optional.of(unionTypeBuilder.build());
      }
    } else {
      return Optional.empty();
    }
  }

  protected static String getCollectionExpression(MessageProcessorNode messageProcessorNode) {
    if (messageProcessorNode == null) {
      return null;
    }

    String parameterContent = messageProcessorNode.getComponentModel().getParameters().get(ATTR_COLLECTION);
    if (parameterContent == null) {
      parameterContent = ComponentModelUtils.collectComponentModelById(messageProcessorNode, PARAMETER_COLLECTION_IDENTIFIER)
          .map(ComponentModel::getTextContent).orElse(null);
    }
    return parameterContent;
  }


  protected static EventType buildInnerEventType(MessageProcessorNode messageProcessorNode, EventType inputEventType,
                                                 TypingMuleAstVisitorContext typingMuleAstVisitorContext,
                                                 String defaultCounterVariableName) {
    final ComponentModel componentModel = messageProcessorNode.getComponentModel();
    final String collectionExpression =
        Optional.ofNullable(getCollectionExpression(messageProcessorNode)).orElse(DEFAULT_COLLECTION_EXPRESSION);

    final boolean batchMode = isBatchMode(componentModel);

    final MuleEventMetadataTypeBuilder muleEventMetadataTypeBuilder = TypesHelper.getMuleEventMetadataTypeBuilder();

    final Optional<MessageMetadataType> messageMetadataType = TypeUtils.getMessageMetadataType(inputEventType);
    Optional<MetadataType> payloadMetadataTypeOptional = messageMetadataType.flatMap(
                                                                                     MessageMetadataType::getPayloadType);

    final AstNotification astNotification = typingMuleAstVisitorContext.getAstNotification();

    final MetadataType sourceCollectionMetadataType = ExpressionLanguageUtils.extractExpression(collectionExpression)
        .map(e -> ExpressionLanguageUtils
            .resolveExpressionType(e, inputEventType,
                                   typingMuleAstVisitorContext.getTypeBindings(),
                                   typingMuleAstVisitorContext.getExpressionLanguageMetadataService(),
                                   new ExpressionLanguageMetadataService.MessageCallback() {

                                     @Override
                                     public void warning(String message,
                                                         ExpressionLanguageMetadataService.MessageLocation messageLocation) {
                                       astNotification
                                           .reportWarning(messageProcessorNode.getAstNodeLocation(),
                                                          NotificationMessages.MSG_SCRIPTING_LANGUAGE_WARNING(e, message));
                                     }

                                     @Override
                                     public void error(String message,
                                                       ExpressionLanguageMetadataService.MessageLocation messageLocation) {
                                       astNotification
                                           .reportError(messageProcessorNode.getAstNodeLocation(),
                                                        NotificationMessages.MSG_SCRIPTING_LANGUAGE_ERROR(e, message));
                                     }

                                   }))
        .orElse(null);
    MetadataType elementMetadataType =
        buildInnerPayloadType(sourceCollectionMetadataType).orElse(TypesHelper.getTypeBuilder().anyType().build());

    if (batchMode) {
      muleEventMetadataTypeBuilder.message().payload().arrayType().of(elementMetadataType);
      muleEventMetadataTypeBuilder.message().attributes(TypesHelper.getTypeBuilder().voidType().build());
    } else {
      final Optional<MessageMetadataType> optionalElementMessageMetadataType =
          TypeUtils.asMessageMetadataType(elementMetadataType);
      if (optionalElementMessageMetadataType.isPresent()) {
        final MessageMetadataType elementMessageMetadataType = optionalElementMessageMetadataType.get();
        elementMessageMetadataType.getPayloadType().ifPresent(metadataType -> {
          muleEventMetadataTypeBuilder.message().payload(metadataType);
        });
        muleEventMetadataTypeBuilder.message()
            .attributes(elementMessageMetadataType.getAttributesType().orElse(TypesHelper.getTypeBuilder().voidType().build()));
      } else {
        muleEventMetadataTypeBuilder.message().payload(elementMetadataType);
        muleEventMetadataTypeBuilder.message().attributes(TypesHelper.getTypeBuilder().voidType().build());
      }
    }

    String counterVariableName =
        Optional.ofNullable(componentModel.getParameters().get(ATTR_COUNTER_VARIABLE_NAME))
            .orElse(defaultCounterVariableName);
    muleEventMetadataTypeBuilder.addVariable(counterVariableName).numberType();

    String messageVariableName =
        Optional.ofNullable(componentModel.getParameters().get(ATTR_MESSAGE_VARIABLE_NAME))
            .orElse(DEFAULT_MESSAGE_VARIABLE_NAME);
    muleEventMetadataTypeBuilder.addVariable(messageVariableName,
                                             messageMetadataType
                                                 .orElse(TypesHelper.getMessageMetadataTypeBuilder().build()));


    final MuleEventMetadataType muleEventMetadataType = muleEventMetadataTypeBuilder.build();
    return TypeUtils.asEventType(muleEventMetadataType);
  }

  private static String getDefaultMessageVariableName() {
    return DEFAULT_MESSAGE_VARIABLE_NAME;
  }

  private static String getDefaultCounterVariableName() {
    return DEFAULT_COUNTER_VARIABLE_NAME;
  }

  private EventType buildExpectedEventType(MessageProcessorNode messageProcessorNode, EventType innerEventType) {
    final ComponentModel componentModel = messageProcessorNode.getComponentModel();
    final String collectionExpression =
        Optional.ofNullable(getCollectionExpression(messageProcessorNode)).orElse(DEFAULT_COLLECTION_EXPRESSION);

    final boolean batchMode = isBatchMode(componentModel);

    final MuleEventMetadataTypeBuilder muleEventMetadataTypeBuilder = TypesHelper.getMuleEventMetadataTypeBuilder();

    if (collectionExpression.equals(DEFAULT_COLLECTION_EXPRESSION)) {
      MetadataType payloadMetadataType = TypeUtils.getMessageMetadataType(innerEventType).flatMap(
                                                                                                  MessageMetadataType::getPayloadType)
          .orElse(null);
      if (payloadMetadataType != null) {
        if (batchMode) {
          muleEventMetadataTypeBuilder.message().payload(payloadMetadataType);
        } else {
          muleEventMetadataTypeBuilder.message()
              .payload(TypesHelper.getTypeBuilder().arrayType().of(payloadMetadataType).build());
        }
      } else {
        muleEventMetadataTypeBuilder.message().payload().arrayType().of().anyType();
      }
    }

    final MuleEventMetadataType muleEventMetadataType = muleEventMetadataTypeBuilder.build();
    return TypeUtils.asEventType(muleEventMetadataType);
  }


  @Override
  protected EventType resolve(MessageProcessorNode messageProcessorNode, EventType inputEventType,
                              TypingMuleAstVisitor typingMuleAstVisitor,
                              TypingMuleAstVisitorContext typingMuleAstVisitorContext) {
    //    final EventType useEventType = buildInnerEventType(messageProcessorNode, inputEventType, typingMuleAstVisitorContext);
    //    EventType innerEventType = TypeUtils.merge(inputEventType, useEventType);
    EventType innerEventType = inputEventType;

    List<MessageProcessorNode> messageProcessorNodes =
        messageProcessorNode.getMessageProcessorNodes().collect(Collectors.toList());

    Optional<EventType> previousEventType = Optional.empty();
    for (MessageProcessorNode pipedMessageProcessorNode : messageProcessorNodes) {
      EventType pipedInputEvent;

      if (previousEventType.isPresent()) {
        pipedInputEvent = previousEventType.map(eventType -> eventType).orElse(new EventType());
      } else {
        pipedInputEvent = innerEventType;
      }

      EventType eventType =
          typingMuleAstVisitor.resolveType(pipedMessageProcessorNode, pipedInputEvent, typingMuleAstVisitorContext);
      previousEventType = Optional.of(eventType);
    }

    return resolveOutputEventType(messageProcessorNode, inputEventType, previousEventType.orElse(null), typingMuleAstVisitor,
                                  typingMuleAstVisitorContext);
  }

  protected EventType resolveOutputEventType(MessageProcessorNode messageProcessorNode, EventType inputEventType,
                                             EventType innerEventType, TypingMuleAstVisitor typingMuleAstVisitor,
                                             TypingMuleAstVisitorContext typingMuleAstVisitorContext) {
    final EventType outputEventType = inputEventType;

    //    messageProcessorNode.annotate(new UsesTypeAnnotation(new EventType()));
    messageProcessorNode.annotate(new DefinesTypeAnnotation(new EventType()));

    return outputEventType;
  }

  @Override
  protected EventType resolveExpectedInputEventType(EventType innerEventType,
                                                    MessageProcessorNode messageProcessorNode,
                                                    ExpectedAstVisitor expectedAstVisitor,
                                                    ExpectedAstVisitorContext visitorContext) {
    return buildExpectedEventType(messageProcessorNode, innerEventType);
  }

  @Override
  protected boolean isPropagates(MessageProcessorNode messageProcessorNode) {
    return true;
  }

  @Override
  public Optional<ComponentModelType> getComponentModelType() {
    return Optional.of(ComponentModelType.MESSAGE_PROCESSOR_NODE);
  }

  @Override
  public Optional<MuleAstParseProvider> getParseProvider() {
    return Optional.of(
                       (componentIdentifier, componentModel, componentModelType, messageProcessorNodeBuilders,
                        muleAstParserContext) -> {
                         return generateAst(componentIdentifier, componentModel, componentModelType,
                                            messageProcessorNodeBuilders);
                       });
  }

  private Optional<AstNodeBuilder> generateAst(ComponentIdentifier componentIdentifier, ComponentModel componentModel,
                                               ComponentModelType componentModelType,
                                               List<MessageProcessorNodeBuilder> messageProcessorNodeBuilders) {
    MessageProcessorNodeBuilder messageProcessorNodeBuilder = new MessageProcessorNodeBuilder(componentIdentifier);
    messageProcessorNodeBuilder.config(componentModel);
    messageProcessorNodeBuilder.componentModelType(componentModelType);
    messageProcessorNodeBuilder.messageProcessor(getComponentIdentifierForeachScopeIn(), m -> {
      m.config(componentModel);
      m.synthetic();
    });
    messageProcessorNodeBuilders
        .forEach(messageProcessorNodeBuilder::messageProcessor);

    return Optional.of(messageProcessorNodeBuilder);
  }

  protected ComponentIdentifier getComponentIdentifierForeachScopeIn() {
    return COMPONENT_IDENTIFIER_FOREACH_SCOPE_IN;
  }

  protected boolean isScope() {
    return true;
  }

  public static class ForEachScopeIn extends SingleNodeTypeResolver {

    @Override
    protected EventType resolve(MessageProcessorNode messageProcessorNode, EventType inputEventType,
                                TypingMuleAstVisitor typingMuleAstVisitor, TypingMuleAstVisitorContext visitorContext) {
      final EventType scopeEventType =
          buildInnerEventType(messageProcessorNode, inputEventType, visitorContext, getDefaultCounterVariableName());
      messageProcessorNode.annotate(new UsesTypeAnnotation(new EventType()));
      messageProcessorNode.annotate(new DefinesTypeAnnotation(scopeEventType));
      return scopeEventType;
    }

    protected String getDefaultCounterVariableName() {
      return DEFAULT_COUNTER_VARIABLE_NAME;
    }

    @Override
    public EventType generateExpected(MessageProcessorNode messageProcessorNode, ExpectedAstVisitor expectedAstVisitor,
                                      ExpectedAstVisitorContext visitorContext) {
      return visitorContext.getExpectedEventType();
    }

    @Override
    protected boolean isPropagates(MessageProcessorNode messageProcessorNode) {
      return true;
    }
  }
}
