/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.datasense.impl;

import static java.util.Optional.empty;
import static org.mule.datasense.impl.phases.builder.MuleAstParser.*;

import org.mule.datasense.api.AnalysisResult;
import org.mule.datasense.api.ComponentResolutionScope;
import org.mule.datasense.api.DataSense;
import org.mule.datasense.api.DataSenseComponentInfo;
import org.mule.datasense.api.DataSenseElementInfo;
import org.mule.datasense.api.DataSenseInfo;
import org.mule.datasense.api.DataSenseResolutionScope;
import org.mule.datasense.api.metadataprovider.ApplicationModel;
import org.mule.datasense.api.metadataprovider.DataSenseProvider;
import org.mule.datasense.extension.DataSenseModuleRegistry;
import org.mule.datasense.impl.model.annotations.ComponentLocationAnnotation;
import org.mule.datasense.impl.model.annotations.DefinesTypeAnnotation;
import org.mule.datasense.impl.model.annotations.ExpectedEventAnnotation;
import org.mule.datasense.impl.model.annotations.IncomingEventAnnotation;
import org.mule.datasense.impl.model.annotations.UsesTypeAnnotation;
import org.mule.datasense.impl.model.ast.AstNotification;
import org.mule.datasense.impl.model.ast.MuleApplicationNode;
import org.mule.datasense.impl.model.types.EventType;
import org.mule.datasense.impl.model.types.TypeUtils;
import org.mule.datasense.impl.phases.typing.resolver.DataSenseTypeResolverRegistry;
import org.mule.datasense.impl.phases.typing.resolver.FlowTypeResolver;
import org.mule.datasense.impl.phases.typing.resolver.TypeResolverRegistry;
import org.mule.datasense.impl.tooling.MetadataQuery;
import org.mule.datasense.impl.tooling.MetadataQueryResult;
import org.mule.datasense.impl.tooling.NodeInfoQuery;
import org.mule.datasense.impl.tooling.TypeContextQuery;
import org.mule.datasense.impl.tooling.TypeContextQueryResult;
import org.mule.datasense.impl.tooling.TypeInfoQuery;
import org.mule.datasense.impl.tooling.TypeInfoQueryResult;
import org.mule.datasense.impl.tooling.TypedApplicationQuery;
import org.mule.datasense.impl.tooling.TypedApplicationQueryResult;
import org.mule.datasense.impl.util.ComponentIdentifierUtils;
import org.mule.datasense.impl.util.LocationUtils;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.location.Location;
import org.mule.runtime.api.i18n.I18nMessageFactory;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 *
 */
public class DefaultDataSense implements DataSense {

  public static final ComponentIdentifier COMPONENT_IDENTIFIER_MULE =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "mule");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_FLOW =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "flow");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_SUBFLOW =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "sub-flow");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_FLOW_SCOPE_IN =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "flow#scope_in");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_FLOW_SCOPE_OUT =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "flow#scope_out");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_LOGGER =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "logger");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_COLLECTION_SPLITTER =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "collection-splitter");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_SCHEDULER =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "scheduler");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_TRANSFORM =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_EE_CORE_PREFIX, MULE_EE_CORE, "transform");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_OPERATION_CALL =
      ComponentIdentifierUtils.createFromNamespaceAndName("internal", "operation-call");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_UNKNOWN_PROCESSOR =
      ComponentIdentifierUtils.createFromNamespaceAndName("internal", "unknown-processor");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_SET_PAYLOAD_ATTRIBUTES =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "set-payload-attributes");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_SET_PAYLOAD =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "set-payload");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_SET_VARIABLE =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "set-variable");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_REMOVE_VARIABLE =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "remove-variable");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_PROCESSOR_CHAIN =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "processor-chain");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_CONTAINER =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "#container");
  // Router: Choice
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_CHOICE =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "choice");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_WHEN =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "when");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_OTHERWISE =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "otherwise");
  // Router: Enricher
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_ENRICHER =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "enricher");
  // Router: ForEach
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_FOREACH =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "foreach");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_FOREACH_SCOPE_IN =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "foreach#scope_in");
  // Router: ParallelForEach
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_PARALLEL_FOREACH =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "parallel-foreach");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_PARALLEL_FOREACH_SCOPE_IN =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "parallel-foreach#scope_in");
  // Router: Scatter-Gather
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_SCATTER_GATHER =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "scatter-gather");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_ROUTE =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "route");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_FLOW_REF =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "flow-ref");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_TRY =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "try");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_ERROR_HANDLER =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "error-handler");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_GLOBAL_ERROR_HANDLER =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "global-error-handler");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_ON_ERROR_CONTINUE =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "on-error-continue");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_ON_ERROR_PROPAGATE =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "on-error-propagate");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_ON_ERROR_SCOPE_IN =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "on-error#scope_in");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_PARSE_TEMPLATE =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "parse-template");
  // Other Scopes
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_FIRST_SUCCESSFUL =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "first-successful");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_UNTIL_SUCCESSFUL =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "until-successful");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_UNTIL_SUCCESSFUL_SCOPE_IN =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "until-successful#scope_in");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_UNTIL_SUCCESSFUL_SCOPE_OUT =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "until-successful#scope_out");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_ROUND_ROBIN =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "round-robin");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_ASYNC =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "async");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_ASYNC_SCOPE_IN =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "async#scope_in");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_ASYNC_SCOPE_OUT =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "async#scope_out");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_EE_CACHE =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_EE_CORE_PREFIX, MULE_EE_CORE, "cache");
  // Batch
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_BATCH_JOB =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_EE_BATCH_PREFIX, MULE_EE_CORE, "job");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_BATCH_JOB_SCOPE_IN =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_EE_BATCH_PREFIX, MULE_EE_CORE, "job#scope_in");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_BATCH_JOB_SCOPE_OUT =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_EE_BATCH_PREFIX, MULE_EE_CORE, "job#scope_out");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_BATCH_PROCESS_RECORDS =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_EE_BATCH_PREFIX, MULE_EE_CORE, "process-records");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_BATCH_ON_COMPLETE =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_EE_BATCH_PREFIX, MULE_EE_CORE, "on-complete");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_BATCH_ON_COMPLETE_SCOPE_IN =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_EE_BATCH_PREFIX, MULE_EE_CORE, "on-complete#scope_in");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_BATCH_ON_COMPLETE_SCOPE_OUT =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_EE_BATCH_PREFIX, MULE_EE_CORE, "on-complete#scope_out");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_BATCH_STEP =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_EE_BATCH_PREFIX, MULE_EE_CORE, "step");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_BATCH_AGGREGATOR =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_EE_BATCH_PREFIX, MULE_EE_CORE, "aggregator");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_BATCH_AGGREGATOR_SCOPE_IN =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_EE_BATCH_PREFIX, MULE_EE_CORE, "aggregator#scope_in");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_BATCH_AGGREGATOR_SCOPE_OUT =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_EE_BATCH_PREFIX, MULE_EE_CORE, "aggregator#scope_out");
  // MUnit
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_MUNIT_TEST =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_MUNIT_PREFIX, MULE_MUNIT, "test");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_MUNIT_TEST_SCOPE_IN =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_MUNIT_PREFIX, MULE_MUNIT, "test#scope_in");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_MUNIT_TEST_SCOPE_OUT =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_MUNIT_PREFIX, MULE_MUNIT, "test#scope_out");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_MUNIT_TEST_BEHAVIOR =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_MUNIT_PREFIX, MULE_MUNIT, "behavior");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_MUNIT_TEST_EXECUTION =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_MUNIT_PREFIX, MULE_MUNIT, "execution");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_MUNIT_TEST_VALIDATION =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_MUNIT_PREFIX, MULE_MUNIT, "validation");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_MUNIT_BEFORE_SUITE =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_MUNIT_PREFIX, MULE_MUNIT, "before-suite");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_MUNIT_BEFORE_SUITE_SCOPE_IN =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_MUNIT_PREFIX, MULE_MUNIT, "before-suite#scope_in");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_MUNIT_BEFORE_SUITE_SCOPE_OUT =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_MUNIT_PREFIX, MULE_MUNIT, "before-suite#scope_out");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_MUNIT_AFTER_SUITE =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_MUNIT_PREFIX, MULE_MUNIT, "after-suite");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_MUNIT_AFTER_SUITE_SCOPE_IN =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_MUNIT_PREFIX, MULE_MUNIT, "after-suite#scope_in");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_MUNIT_AFTER_SUITE_SCOPE_OUT =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_MUNIT_PREFIX, MULE_MUNIT, "after-suite#scope_out");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_MUNIT_BEFORE_TEST =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_MUNIT_PREFIX, MULE_MUNIT, "before-test");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_MUNIT_BEFORE_TEST_SCOPE_IN =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_MUNIT_PREFIX, MULE_MUNIT, "before-test#scope_in");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_MUNIT_BEFORE_TEST_SCOPE_OUT =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_MUNIT_PREFIX, MULE_MUNIT, "before-test#scope_out");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_MUNIT_AFTER_TEST =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_MUNIT_PREFIX, MULE_MUNIT, "after-test");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_MUNIT_AFTER_TEST_SCOPE_IN =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_MUNIT_PREFIX, MULE_MUNIT, "after-test#scope_in");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_MUNIT_AFTER_TEST_SCOPE_OUT =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_MUNIT_PREFIX, MULE_MUNIT, "after-test#scope_out");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_MUNIT_SET_EVENT =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_MUNIT_PREFIX, MULE_MUNIT, "set-event");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_MUNIT_SET_NULL_PAYLOAD =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_MUNIT_PREFIX, MULE_MUNIT, "set-null-payload");
  // Aggregators
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_AGGREGATORS_AGGREGATOR_LISTENER =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_AGGREGATORS_PREFIX, MULE_AGGREGATORS, "aggregator-listener");

  public final static ComponentIdentifier COMPONENT_IDENTIFIER_AGGREGATORS_TIME_BASED_AGGREGATOR =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_AGGREGATORS_PREFIX, MULE_AGGREGATORS, "time-based-aggregator");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_AGGREGATORS_GROUP_BASED_AGGREGATOR =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_AGGREGATORS_PREFIX, MULE_AGGREGATORS, "group-based-aggregator");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_AGGREGATORS_SIZE_BASED_AGGREGATOR =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_AGGREGATORS_PREFIX, MULE_AGGREGATORS, "size-based-aggregator");

  public final static ComponentIdentifier COMPONENT_IDENTIFIER_AGGREGATORS_INCREMENTAL_AGGREGATION =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_AGGREGATORS_PREFIX, MULE_AGGREGATORS, "incremental-aggregation");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_AGGREGATORS_AGGREGATION_COMPLETE =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_AGGREGATORS_PREFIX, MULE_AGGREGATORS, "aggregation-complete");

  public final static ComponentIdentifier COMPONENT_IDENTIFIER_AGGREGATORS_ROUTE_SCOPE_IN =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_AGGREGATORS_PREFIX, MULE_AGGREGATORS,
                                                          "aggregators-route#scope_in");

  public final static ComponentIdentifier COMPONENT_IDENTIFIER_RAISE_ERROR =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_CORE_PREFIX, MULE_CORE, "raise-error");

  public final static ComponentIdentifier COMPONENT_IDENTIFIER_MULE_HTTP_POLICY_PROXY =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_HTTP_POLICY_PREFIX, MULE_HTTP_POLICY, "proxy");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_MULE_HTTP_POLICY_PROXY_SCOPE_IN =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_HTTP_POLICY_PREFIX, MULE_HTTP_POLICY, "proxy#scope_in");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_MULE_HTTP_POLICY_PROXY_SCOPE_OUT =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_HTTP_POLICY_PREFIX, MULE_HTTP_POLICY, "proxy#scope_out");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_MULE_HTTP_POLICY_SOURCE =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_HTTP_POLICY_PREFIX, MULE_HTTP_POLICY, "source");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_MULE_HTTP_POLICY_SOURCE_SCOPE_IN =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_HTTP_POLICY_PREFIX, MULE_HTTP_POLICY, "source#scope_in");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_MULE_HTTP_POLICY_SOURCE_SCOPE_OUT =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_HTTP_POLICY_PREFIX, MULE_HTTP_POLICY, "source#scope_out");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_MULE_HTTP_POLICY_OPERATION =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_HTTP_POLICY_PREFIX, MULE_HTTP_POLICY, "operation");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_MULE_HTTP_POLICY_OPERATION_SCOPE_IN =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_HTTP_POLICY_PREFIX, MULE_HTTP_POLICY, "operation#scope_in");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_MULE_HTTP_POLICY_OPERATION_SCOPE_OUT =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_HTTP_POLICY_PREFIX, MULE_HTTP_POLICY, "operation#scope_out");
  public final static ComponentIdentifier COMPONENT_IDENTIFIER_MULE_HTTP_POLICY_EXECUTE_NEXT =
      ComponentIdentifierUtils.createFromNamespaceAndName(MULE_HTTP_POLICY_PREFIX, MULE_HTTP_POLICY, "execute-next");

  private static final String BUILD_ID = "4.3.0-SNAPSHOT_ac26cb8fdb939afac41313ae9947f8a35c3eb952";

  private static final transient Logger logger = LoggerFactory.getLogger(DefaultDataSense.class);
  private final DataSenseModuleRegistry dataSenseModuleRegistry;

  public DefaultDataSense() {
    dataSenseModuleRegistry = new DataSenseModuleRegistry();
  }

  public static Logger getLogger() {
    return logger;
  }

  private DataSenseModuleRegistry getDataSenseModuleRegistry() {
    return dataSenseModuleRegistry;
  }

  protected TypeResolverRegistry createTypeResolverRegistry() {
    return new DataSenseTypeResolverRegistry();
  }

  /**
   * @param location
   * @param applicationModel
   * @param dataSenseProvider
   * @return
   */
  @Override
  public Optional<DataSenseInfo> resolve(Location location, ApplicationModel applicationModel,
                                         DataSenseProvider dataSenseProvider) {
    return resolve(new ComponentResolutionScope(location), applicationModel, dataSenseProvider);
  }

  /**
   * @param dataSenseResolutionScope
   * @param applicationModel
   * @param dataSenseProvider
   * @return
   */
  @Override
  public Optional<DataSenseInfo> resolve(DataSenseResolutionScope dataSenseResolutionScope, ApplicationModel applicationModel,
                                         DataSenseProvider dataSenseProvider) {
    AstNotification astNotification = new AstNotification();
    astNotification
        .reportInfo(null,
                    I18nMessageFactory.createStaticMessage(
                                                           "Resolving datasense info for component path %s. (build: %s)",
                                                           dataSenseResolutionScope, BUILD_ID));
    DataSenseApplicationModel dataSenseApplicationModel =
        new DataSenseApplicationModel(dataSenseResolutionScope, applicationModel, dataSenseProvider,
                                      createTypeResolverRegistry(),
                                      astNotification);
    dataSenseApplicationModel.build();
    dataSenseApplicationModel.resolve();

    final MuleApplicationNode muleApplicationNode = dataSenseApplicationModel.getMuleApplicationNode();
    final DataSenseProviderResolver dataSenseProviderResolver = dataSenseApplicationModel.getDataSenseProviderResolver();
    Map<Location, DataSenseElementInfo> dataSenseComponentInfoMap = new HashMap<>();

    Optional<Location> dataSenseLocation = empty();
    if (dataSenseResolutionScope instanceof ComponentResolutionScope) {
      final ComponentResolutionScope componentResolutionScope = (ComponentResolutionScope) dataSenseResolutionScope;
      final Location location = componentResolutionScope.getLocation();
      if (!LocationUtils.isFlow(location)) {
        dataSenseLocation = Optional.of(location);
      }
    }

    if (!dataSenseLocation.isPresent()) {
      //      final String flow = location.getGlobalElementName();
      //      final MuleFlowNode muleFlowNode =
      //          muleApplicationNode.getMuleFlowNodes().filter(f -> f.getName().equals(flow)).findFirst()
      //              .orElseThrow(() -> new RuntimeException(String.format(
      //                                                                    "Flow %s not found.", location)));

      muleApplicationNode.findMessageProcessorNodes().forEach(messageProcessorNode -> {
        messageProcessorNode.getAnnotation(ComponentLocationAnnotation.class)
            .map(ComponentLocationAnnotation::getLocation).ifPresent(innerLocation -> {
              perform(new NodeInfoQuery(innerLocation), muleApplicationNode,
                      dataSenseProviderResolver,
                      astNotification).ifPresent(nodeInfoQueryResult -> {

                        // type info
                        Optional<TypeInfoQueryResult> typeInfoQueryResult =
                            perform(new TypeInfoQuery(innerLocation), muleApplicationNode, dataSenseProviderResolver,
                                    astNotification);

                        // type context
                        Optional<TypeContextQueryResult> typeContextQueryResult =
                            perform(new TypeContextQuery(innerLocation), muleApplicationNode, dataSenseProviderResolver,
                                    astNotification);

                        dataSenseComponentInfoMap.put(innerLocation, new DefaultDataSenseInfo(
                                                                                              innerLocation,
                                                                                              null,
                                                                                              null,
                                                                                              typeInfoQueryResult
                                                                                                  .flatMap(TypeInfoQueryResult::getInput)
                                                                                                  .orElse(null),
                                                                                              typeInfoQueryResult
                                                                                                  .flatMap(TypeInfoQueryResult::getOutput)
                                                                                                  .orElse(null),
                                                                                              typeContextQueryResult
                                                                                                  .flatMap(TypeContextQueryResult::getIncoming)
                                                                                                  .orElse(null),
                                                                                              typeContextQueryResult
                                                                                                  .flatMap(TypeContextQueryResult::getExpected)
                                                                                                  .orElse(null),
                                                                                              typeContextQueryResult
                                                                                                  .flatMap(TypeContextQueryResult::getResult)
                                                                                                  .orElse(null),
                                                                                              typeContextQueryResult
                                                                                                  .flatMap(TypeContextQueryResult::getExpectedInput)
                                                                                                  .orElse(null),
                                                                                              null,
                                                                                              astNotification
                                                                                                  .getDataSenseNotifications(innerLocation),
                                                                                              null));
                      });
            });
      });
      muleApplicationNode.getMuleFlowNodes().forEach(muleFlowNode -> {
        muleFlowNode.getAnnotation(ComponentLocationAnnotation.class)
            .map(ComponentLocationAnnotation::getLocation).ifPresent(innerLocation -> {
              muleFlowNode.getRootMessageProcessorNode().ifPresent(rootMessageProcessorNode -> {
                Optional<EventType> expectedEventType =
                    FlowTypeResolver.getScopeInMessageProcessorNode(rootMessageProcessorNode)
                        .flatMap(scopeInMessageProcessorNode -> {
                          return scopeInMessageProcessorNode.getAnnotation(ExpectedEventAnnotation.class).map(
                                                                                                              ExpectedEventAnnotation::getEventType);
                        });

                Optional<EventType> incomingEventType =
                    FlowTypeResolver.getScopeOutMessageProcessorNode(rootMessageProcessorNode)
                        .flatMap(scopeOutMessageProcessorNode -> {
                          return scopeOutMessageProcessorNode.getAnnotation(IncomingEventAnnotation.class).map(
                                                                                                               IncomingEventAnnotation::getEventType);
                        });

                dataSenseComponentInfoMap.put(innerLocation, new DefaultDataSenseInfo(
                                                                                      innerLocation,
                                                                                      null,
                                                                                      null,
                                                                                      TypeUtils
                                                                                          .asMuleEventMetadataType(muleFlowNode
                                                                                              .getAnnotation(UsesTypeAnnotation.class)
                                                                                              .map(UsesTypeAnnotation::getUsesEventType)
                                                                                              .orElse(new EventType()))
                                                                                          .build(),
                                                                                      TypeUtils
                                                                                          .asMuleEventMetadataType(muleFlowNode
                                                                                              .getAnnotation(DefinesTypeAnnotation.class)
                                                                                              .map(DefinesTypeAnnotation::getDefinesEventType)
                                                                                              .orElse(new EventType()))
                                                                                          .build(),
                                                                                      TypeUtils
                                                                                          .asMuleEventMetadataType(incomingEventType
                                                                                              .orElse(new EventType()))
                                                                                          .build(),
                                                                                      TypeUtils
                                                                                          .asMuleEventMetadataType(expectedEventType
                                                                                              .orElse(new EventType()))
                                                                                          .build(),
                                                                                      null,
                                                                                      null,
                                                                                      null,
                                                                                      astNotification
                                                                                          .getDataSenseNotifications(innerLocation),
                                                                                      null));
              });

            });
      });

      return Optional.of(new DefaultDataSenseInfo(
                                                  null,
                                                  null,
                                                  null,
                                                  null,
                                                  null,
                                                  null,
                                                  null,
                                                  null,
                                                  null, astNotification.getPlainMessages(),
                                                  astNotification.getDataSenseNotifications(), dataSenseComponentInfoMap,
                                                  TypeInfoQuery.getGlobalTypeBindings(muleApplicationNode),
                                                  TypeInfoQuery.getFunctionBindings(muleApplicationNode)));
    } else {
      Location location = dataSenseLocation.get();
      return perform(new NodeInfoQuery(location), muleApplicationNode, dataSenseProviderResolver,
                     astNotification).map(nodeInfoQueryResult -> {
                       // type info
                       Optional<TypeInfoQueryResult> typeInfoQueryResult =
                           perform(new TypeInfoQuery(location), muleApplicationNode, dataSenseProviderResolver,
                                   astNotification);

                       // type context
                       Optional<TypeContextQueryResult> typeContextQueryResult =
                           perform(new TypeContextQuery(location), muleApplicationNode, dataSenseProviderResolver,
                                   astNotification);

                       // metadata
                       // dynamic metadata
                       Optional<MetadataQueryResult> metadataQueryResult =
                           perform(new MetadataQuery(location), muleApplicationNode, dataSenseProviderResolver,
                                   astNotification);

                       dataSenseComponentInfoMap.put(location, new DefaultDataSenseInfo(
                                                                                        location,
                                                                                        null,
                                                                                        null,
                                                                                        typeInfoQueryResult
                                                                                            .flatMap(TypeInfoQueryResult::getInput)
                                                                                            .orElse(null),
                                                                                        typeInfoQueryResult
                                                                                            .flatMap(TypeInfoQueryResult::getOutput)
                                                                                            .orElse(null),
                                                                                        typeContextQueryResult
                                                                                            .flatMap(TypeContextQueryResult::getIncoming)
                                                                                            .orElse(null),
                                                                                        typeContextQueryResult
                                                                                            .flatMap(TypeContextQueryResult::getExpected)
                                                                                            .orElse(null),
                                                                                        typeContextQueryResult
                                                                                            .flatMap(TypeContextQueryResult::getResult)
                                                                                            .orElse(null),
                                                                                        typeContextQueryResult
                                                                                            .flatMap(TypeContextQueryResult::getExpectedInput)
                                                                                            .orElse(null),
                                                                                        null,
                                                                                        astNotification
                                                                                            .getDataSenseNotifications(location),
                                                                                        null));

                       return new DefaultDataSenseInfo(
                                                       location,
                                                       metadataQueryResult.flatMap(MetadataQueryResult::getOperationModel)
                                                           .orElse(null),
                                                       metadataQueryResult.flatMap(MetadataQueryResult::getSourceModel)
                                                           .orElse(null),
                                                       typeInfoQueryResult.flatMap(TypeInfoQueryResult::getInput).orElse(null),
                                                       typeInfoQueryResult.flatMap(TypeInfoQueryResult::getOutput).orElse(null),
                                                       typeContextQueryResult.flatMap(TypeContextQueryResult::getIncoming)
                                                           .orElse(null),
                                                       typeContextQueryResult.flatMap(TypeContextQueryResult::getExpected)
                                                           .orElse(null),
                                                       typeContextQueryResult.flatMap(TypeContextQueryResult::getResult)
                                                           .orElse(null),
                                                       typeContextQueryResult.flatMap(TypeContextQueryResult::getExpectedInput)
                                                           .orElse(null),
                                                       astNotification.getPlainMessages(),
                                                       astNotification.getDataSenseNotifications(), dataSenseComponentInfoMap,
                                                       TypeInfoQuery.getGlobalTypeBindings(muleApplicationNode),
                                                       TypeInfoQuery.getFunctionBindings(muleApplicationNode));
                     });
    }
  }

  @Override
  public Optional<DataSenseComponentInfo> resolveComponent(Location location, ApplicationModel applicationModel,
                                                           DataSenseProvider dataSenseProvider) {
    AstNotification astNotification = new AstNotification();
    astNotification
        .reportInfo(null,
                    I18nMessageFactory.createStaticMessage(
                                                           "Resolving datasense info for component path %s. (build: %s) ",
                                                           location, BUILD_ID));

    DataSenseApplicationModel dataSenseApplicationModel =
        new DataSenseApplicationModel(new ComponentResolutionScope(location), applicationModel, dataSenseProvider,
                                      createTypeResolverRegistry(),
                                      astNotification);
    dataSenseApplicationModel.build();
    if (!dataSenseApplicationModel.resolveComponent(location)) {
      return empty();
    } ;

    final MuleApplicationNode muleApplicationNode = dataSenseApplicationModel.getMuleApplicationNode();
    DataSenseProviderResolver dataSenseProviderResolver = dataSenseApplicationModel.getDataSenseProviderResolver();
    return perform(new NodeInfoQuery(location), muleApplicationNode, dataSenseProviderResolver,
                   astNotification).map(nodeInfoQueryResult -> {
                     // metadata
                     // dynamic metadata
                     Optional<MetadataQueryResult> metadataQueryResult =
                         perform(new MetadataQuery(location), muleApplicationNode, dataSenseProviderResolver,
                                 astNotification);

                     return new DefaultDataSenseInfo(
                                                     location,
                                                     metadataQueryResult.flatMap(MetadataQueryResult::getOperationModel)
                                                         .orElse(null),
                                                     metadataQueryResult.flatMap(MetadataQueryResult::getSourceModel)
                                                         .orElse(null),
                                                     null,
                                                     null,
                                                     null,
                                                     null,
                                                     null,
                                                     null, astNotification.getPlainMessages(),
                                                     astNotification.getDataSenseNotifications(), null);
                   });
  }

  /**
   * @param dataSenseProvider
   * @param applicationModel
   * @return
   */
  @Override
  public AnalysisResult analyze(DataSenseProvider dataSenseProvider, ApplicationModel applicationModel) {
    return null;
  }

  private <T extends TypedApplicationQueryResult> Optional<T> perform(TypedApplicationQuery<T> typedApplicationQuery,
                                                                      MuleApplicationNode muleApplicationNode,
                                                                      DataSenseProviderResolver dataSenseProviderResolver,
                                                                      AstNotification astNotification) {
    return typedApplicationQuery.perform(muleApplicationNode, dataSenseProviderResolver, astNotification);
  }
}
