package org.mule.datasense.impl.model.types;

import static java.util.Optional.empty;
import static org.mule.datasense.impl.model.types.TypesHelper.MULE_EVENT_ATTRIBUTES;
import static org.mule.datasense.impl.model.types.TypesHelper.MULE_EVENT_ERROR;
import static org.mule.datasense.impl.model.types.TypesHelper.MULE_EVENT_PAYLOAD;
import org.mule.metadata.api.annotation.TypeIdAnnotation;
import org.mule.metadata.api.builder.ObjectFieldTypeBuilder;
import org.mule.metadata.api.builder.UnionTypeBuilder;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.NullType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.model.SimpleType;
import org.mule.metadata.api.model.UnionType;
import org.mule.metadata.api.model.VoidType;
import org.mule.metadata.message.api.LocationAnnotation;
import org.mule.metadata.message.api.MessageMetadataType;
import org.mule.metadata.message.api.MessageMetadataTypeBuilder;
import org.mule.metadata.message.api.MuleEventMetadataType;
import org.mule.metadata.message.api.MuleEventMetadataTypeBuilder;
import org.mule.metadata.message.api.el.TypeBindings;
import org.mule.runtime.api.message.Message;
import org.mule.runtime.api.metadata.ExpressionLanguageMetadataService;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class TypeUtils {

  public static boolean isAssignable(MetadataType sourceType, MetadataType targetType) {
    return TypeEquivalence.canBeAssignedTo(sourceType, targetType, null);
  }

  public static Stream<VarDecl> asVarDecls(MessageMetadataType messageMetadataType, boolean includeVoids) {
    Set<VarDecl> varDecls = new HashSet<>();
    messageMetadataType.getPayloadType().filter(metadataType -> includeVoids || !(metadataType instanceof VoidType))
        .ifPresent(metadataType -> {
          varDecls.add(new VarDecl(MULE_EVENT_PAYLOAD, metadataType));
        });
    messageMetadataType.getAttributesType().filter(metadataType -> includeVoids || !(metadataType instanceof VoidType))
        .ifPresent(metadataType -> {
          varDecls.add(new VarDecl(TypesHelper.MULE_EVENT_ATTRIBUTES, metadataType));
        });
    return varDecls.stream();
  }

  public static Stream<VarDecl> asVarDecls(MessageMetadataType messageMetadataType) {
    return asVarDecls(messageMetadataType, true);

  }

  public static Stream<VarDecl> asVarDecls(MuleEventMetadataType muleEventMetadataType) {
    Set<VarDecl> varDecls = new HashSet<>();
    if (muleEventMetadataType != null) {
      varDecls.addAll(asVarDecls(muleEventMetadataType.getMessageType()).collect(Collectors.toList()));
      muleEventMetadataType.getErrorType().ifPresent(metadataType -> {
        varDecls.add(new VarDecl(MULE_EVENT_ERROR, metadataType));
      });
      muleEventMetadataType.getVariables().getFields().forEach(objectFieldType -> {
        varDecls.add(new VarDecl(objectFieldType.getKey().getName().getLocalPart(),
                                 objectFieldType.getValue(), !objectFieldType.isRequired()));
      });
    }
    return varDecls.stream();
  }

  private static void enrichWithLocation(VarDecl dataDeclaration, Consumer<LocationAnnotation> locationAnnotationConsumer) {
    dataDeclaration.getMetadata().ifPresent(varDeclMetadata -> {
      varDeclMetadata.getComponentLocation().ifPresent(componentLocation -> {
        locationAnnotationConsumer.accept(new LocationAnnotation(
                                                                 componentLocation
                                                                     .getLocation()/*,
                                                                                   componentLocation.getRootContainerName(),
                                                                                   componentLocation.getFileName().orElse(null),
                                                                                   componentLocation.getLineInFile().orElse(null)*/
        ));
      });
    });
  }

  public static MuleEventMetadataTypeBuilder asMuleEventMetadataType(EventType eventType) {
    MuleEventMetadataTypeBuilder muleEventMetadataTypeBuilder = MuleEventMetadataType.builder();

    if (eventType.getVarDecls() != null) {
      eventType.getVarDecls().forEach(dataDeclaration -> {
        MetadataType metadataType = dataDeclaration.getType();

        Consumer<LocationAnnotation> locationAnnotationConsumer = null;
        if (dataDeclaration.getName().equals(MULE_EVENT_PAYLOAD)) {
          muleEventMetadataTypeBuilder.message().payload(dataDeclaration.getType());
          locationAnnotationConsumer =
              locationAnnotation -> muleEventMetadataTypeBuilder.message().withPayloadAnnotation(locationAnnotation);

        } else if (dataDeclaration.getName().equals(TypesHelper.MULE_EVENT_ATTRIBUTES)) {
          muleEventMetadataTypeBuilder.message().attributes(dataDeclaration.getType());
          locationAnnotationConsumer =
              locationAnnotation -> muleEventMetadataTypeBuilder.message().withAttributesAnnotation(locationAnnotation);
        } else if (dataDeclaration.getName().equals(TypesHelper.MULE_EVENT_ERROR)) {
          muleEventMetadataTypeBuilder.error(dataDeclaration.getType());
        } else {
          final ObjectFieldTypeBuilder variableField =
              muleEventMetadataTypeBuilder.addVariableField(dataDeclaration.getName());
          variableField.value(metadataType);
          variableField.required(!dataDeclaration.isOptional());
          locationAnnotationConsumer = locationAnnotation -> variableField.with(locationAnnotation);
        }
        if (locationAnnotationConsumer != null) {
          enrichWithLocation(dataDeclaration, locationAnnotationConsumer);
        }
      });
    }
    return muleEventMetadataTypeBuilder;
  }

  public static EventType asEventType(MuleEventMetadataType muleEventMetadataType) {
    return new EventType(asVarDecls(muleEventMetadataType));
  }

  public static EventType annotate(EventType eventType, VarDeclMetadata varDeclMetadata) {
    return new EventType(eventType.getVarDecls()
        .map(varDecl -> new VarDecl(varDecl.getName(), varDecl.getType(), varDeclMetadata)));
  }

  public static final Predicate<? super VarDecl> USE_EVENT_PREDICATE = varDecl -> !varDecl.getName().equals(MULE_EVENT_ERROR);

  public static EventType filter(EventType eventType, Predicate<? super VarDecl> predicate) {
    return new EventType(eventType.getVarDecls().filter(predicate));
  }

  public static EventType merge(EventType eventType1, EventType eventType2) {
    if (eventType1 == null) {
      return eventType2;
    } else if (eventType2 == null) {
      return eventType1;
    }

    Set<VarDecl> collect = eventType2.getVarDecls().collect(Collectors.toSet());
    collect.addAll(eventType1.getVarDecls().collect(Collectors.toSet()));
    return new EventType(collect.stream());
  }

  public static EventType minus(EventType eventType1, EventType eventType2) {
    if (eventType1 == null) {
      return null;
    } else if (eventType2 == null) {
      return eventType1;
    }

    return new EventType(eventType1.getVarDecls().filter(varDecl -> !eventType2.get(varDecl.getName()).isPresent()));
  }



  public static MetadataType override(MetadataType metadataType1, MetadataType metadataType2) {
    if (metadataType1 == null) {
      return metadataType2;
    } else if (metadataType2 == null) {
      return metadataType1;
    }

    MetadataType result;

    final MessageMetadataTypeBuilder messageMetadataTypeBuilder = TypesHelper.getMessageMetadataTypeBuilder();

    if (metadataType1 instanceof MessageMetadataType) {
      final MessageMetadataType messageMetadataType1 = (MessageMetadataType) metadataType1;
      if (metadataType2 instanceof MessageMetadataType) {
        final MessageMetadataType messageMetadataType2 = (MessageMetadataType) metadataType2;

        final MetadataType payloadMetadataType =
            messageMetadataType2.getPayloadType().orElse(messageMetadataType1.getPayloadType().orElse(null));
        if (payloadMetadataType != null) {
          messageMetadataTypeBuilder.payload(payloadMetadataType);
        }

        final MetadataType attributesMetadataType =
            messageMetadataType2.getAttributesType().orElse(messageMetadataType1.getAttributesType().orElse(null));
        if (attributesMetadataType != null) {
          messageMetadataTypeBuilder.attributes(attributesMetadataType);
        }

      } else {
        messageMetadataTypeBuilder.payload(metadataType2);
        messageMetadataType1.getAttributesType().ifPresent(messageMetadataTypeBuilder::attributes);
      }
      result = messageMetadataTypeBuilder.build();
    } else {
      result = metadataType2;
    }

    return result;
  }

  public static Optional<Map<String, MetadataType>> unifyTypes(MetadataType metadataType1, MetadataType metadataType2) {
    MetadataTypeUnification metadataTypeUnification = new MetadataTypeUnification();
    return metadataTypeUnification.unify(metadataType1, metadataType2);
  }

  public static MetadataType substitute(MetadataType metadataType, Map<String, MetadataType> substitution) {
    if (substitution == null || substitution.isEmpty()) {
      return metadataType;
    }

    MetadataTypeSubstitution metadataTypeSubstitution = new MetadataTypeSubstitution(substitution);
    return metadataTypeSubstitution.transform(metadataType);
  }

  public static EventType union(List<EventType> eventTypes,
                                ExpressionLanguageMetadataService expressionLanguageMetadataService) {
    if (eventTypes == null || eventTypes.size() == 0) {
      return new EventType();
    }

    Map<String, List<MetadataType>> metadataTypesByVar = new HashMap<>();
    Map<String, Integer> metadataTypesCountByVar = new HashMap<>();
    eventTypes.forEach(eventType -> {
      eventType.getVarDecls().forEach(varDecl -> {
        List<MetadataType> metadataTypes = metadataTypesByVar.get(varDecl.getName());
        int count;
        if (metadataTypes == null) {
          metadataTypes = new ArrayList<>();
          metadataTypesByVar.put(varDecl.getName(), metadataTypes);
          count = 0;
        } else {
          count = metadataTypesCountByVar.get(varDecl.getName());
        }
        metadataTypes.add(varDecl.getType());
        metadataTypesCountByVar.put(varDecl.getName(), count + 1);
      });
    });

    List<VarDecl> varDecls = new ArrayList<>();
    metadataTypesByVar.entrySet().forEach(metadataTypesByVarEntry -> {
      final String varName = metadataTypesByVarEntry.getKey();
      final List<MetadataType> typeList = metadataTypesByVarEntry.getValue();

      final Integer count = metadataTypesCountByVar.get(varName);
      if (count > 0) {

        VarDecl varDecl = null;
        boolean optional = count < eventTypes.size();
        final MetadataType metadataType;
        if (typeList.size() == 1) {
          metadataType = typeList.iterator().next();
        } else {
          metadataType = expressionLanguageMetadataService.unify(typeList);
        }
        varDecl = new VarDecl(varName, metadataType, optional);
        varDecls.add(varDecl);
      }
    });
    return new EventType(varDecls.stream());
  }

  public static EventType intersection(List<EventType> eventTypes,
                                       ExpressionLanguageMetadataService expressionLanguageMetadataService) {
    if (eventTypes == null || eventTypes.size() == 0) {
      return new EventType();
    }

    Map<String, List<MetadataType>> metadataTypesByVar = new HashMap<>();
    eventTypes.forEach(eventType -> {
      eventType.getVarDecls().forEach(varDecl -> {
        List<MetadataType> metadataTypes = metadataTypesByVar.get(varDecl.getName());
        if (metadataTypes == null) {
          metadataTypes = new ArrayList<>();
          metadataTypesByVar.put(varDecl.getName(), metadataTypes);
        }
        metadataTypes.add(varDecl.getType());
      });
    });

    List<VarDecl> varDecls = new ArrayList<>();
    metadataTypesByVar.entrySet().forEach(stringListEntry -> {
      final String varName = stringListEntry.getKey();
      final List<MetadataType> typeList = stringListEntry.getValue();

      if (typeList.size() > 0) {
        VarDecl varDecl = null;
        if (typeList.size() == 1) {
          varDecl = new VarDecl(varName, typeList.iterator().next());
        } else {
          // @todo intersection not being generated until supported
          /*
                  final IntersectionTypeBuilder intersectionTypeBuilder = TypesHelper.getTypeBuilder().intersectionType();
                  typeList.forEach(intersectionTypeBuilder::of);
                  varDecl = new VarDecl(varName, intersectionTypeBuilder.build());
          */

          List<MetadataType> metadataTypeList = new ArrayList<>();
          typeList.forEach(metadataType -> {
            if (metadataType instanceof SimpleType) {
              if (!metadataTypeList.contains(metadataType)) {
                metadataTypeList.add(metadataType);
              }
            } else {
              metadataTypeList.add(metadataType);
            }
          });

          MetadataType metadataType;
          if (metadataTypeList.size() == 1) {
            metadataType = metadataTypeList.iterator().next();
          } else {
            metadataType = expressionLanguageMetadataService.intersect(metadataTypeList);
          }

          varDecl = new VarDecl(varName, metadataType);
        }
        varDecls.add(varDecl);
      }
    });
    return new EventType(varDecls.stream());
  }


  public static EventType override(EventType eventType1, EventType eventType2) {
    if (eventType1 == null) {
      return eventType2;
    } else if (eventType2 == null) {
      return eventType1;
    }

    Map<String, VarDecl> varDecls = new HashMap<>();
    eventType1.getVarDecls().forEach(varDecl -> varDecls.put(varDecl.getName(), varDecl));
    eventType2.getVarDecls().forEach(varDecl -> {
      final VarDecl overridenVarDecl = varDecls.get(varDecl.getName());
      if (overridenVarDecl != null) {
        varDecls.put(overridenVarDecl.getName(),
                     new VarDecl(overridenVarDecl.getName(), override(overridenVarDecl.getType(), varDecl.getType()),
                                 varDecl.isOptional()));
      } else {
        varDecls.put(varDecl.getName(), varDecl); // todo: this is forcing the event type
      }
    });
    return new EventType(varDecls.values().stream());
  }

  public static EventType createEventType(MetadataType metadataType) {
    MuleEventMetadataTypeBuilder muleEventMetadataTypeBuilder = TypesHelper.getMuleEventMetadataTypeBuilder();
    if (metadataType != null) {
      muleEventMetadataTypeBuilder.message().payload(metadataType);
    }
    MuleEventMetadataType muleEventMetadataType = muleEventMetadataTypeBuilder.build();
    return TypeUtils.asEventType(muleEventMetadataType);
  }

  public static Optional<MetadataType> getMessagePayloadType(MetadataType metadataType) {
    if (metadataType == null) {
      return empty();
    }

    return Optional.ofNullable(metadataType instanceof MessageMetadataType
        ? ((MessageMetadataType) metadataType).getPayloadType().orElse(null) : null);
  }

  public static Optional<MetadataType> getMessageAttributesType(MetadataType metadataType) {
    if (metadataType == null) {
      return empty();
    }

    return Optional.ofNullable(metadataType instanceof MessageMetadataType
        ? ((MessageMetadataType) metadataType).getAttributesType().orElse(null) : null);
  }

  public static Optional<MessageMetadataType> getMessageMetadataType(EventType eventType) {
    if (eventType == null) {
      return empty();
    }

    final VarDecl messagePayload = eventType.get(MULE_EVENT_PAYLOAD).orElse(null);
    final VarDecl messageAttributes = eventType.get(MULE_EVENT_ATTRIBUTES).orElse(null);
    if (messagePayload != null || messageAttributes != null) {
      final MessageMetadataTypeBuilder messageMetadataTypeBuilder = TypesHelper.getMessageMetadataTypeBuilder();
      eventType.get(MULE_EVENT_PAYLOAD).ifPresent(varDecl -> {
        messageMetadataTypeBuilder.payload(varDecl.getType());
      });
      eventType.get(MULE_EVENT_ATTRIBUTES).ifPresent(varDecl -> {
        messageMetadataTypeBuilder.attributes(varDecl.getType());
      });
      return Optional.of(messageMetadataTypeBuilder.build());
    } else {
      return empty();
    }
  }

  private static boolean isMessageMetadataType(MetadataType elementMetadataType) {
    return elementMetadataType.getAnnotation(TypeIdAnnotation.class)
        .map(typeIdAnnotation -> Message.class.getName().equals(typeIdAnnotation.getValue())).orElse(false);
  }

  public static MessageMetadataType asMessageMetadataTypeOrEmptyMessage(MessageMetadataType messageMetadataType) {
    return TypeUtils.asMessageMetadataType(messageMetadataType).orElse(
                                                                       TypesHelper
                                                                           .getMessageMetadataTypeBuilder()
                                                                           .build());
  }

  public static Optional<MessageMetadataType> asMessageMetadataType(MetadataType metadataType) {
    Optional<MessageMetadataType> result = empty();
    if (metadataType instanceof ObjectType && isMessageMetadataType(metadataType)) {
      result = Optional.of(MessageMetadataType.builder((ObjectType) metadataType)).map(MessageMetadataTypeBuilder::build);
    }
    return result;
  }

  public static Optional<MuleEventMetadataType> asMuleEventMetadataType(MetadataType metadataType) {
    Optional<MuleEventMetadataType> result = empty();
    if (metadataType instanceof ObjectType) {
      result = Optional.of(MuleEventMetadataType.builder((ObjectType) metadataType)).map(MuleEventMetadataTypeBuilder::build);
    }
    return result;
  }

  public static TypeBindings buildTypeBindings(MuleEventMetadataType muleEventMetadataType, TypeBindings typeBindings) {
    final TypeBindings.Builder builder = TypeBindings.builder();
    builder.addAll(typeBindings);
    builder.addAll(TypeBindings.builder(muleEventMetadataType).build());
    return builder.build();
  }

  public static MetadataType removeNullsFromUnionMetadataType(MetadataType metadataType) {
    if (!(metadataType instanceof UnionType)) {
      return metadataType;
    }

    final UnionType unionType = (UnionType) metadataType;
    final List<MetadataType> metadataTypes =
        unionType.getTypes().stream().filter(elementMetadataType -> !(elementMetadataType instanceof NullType)).collect(
                                                                                                                        Collectors
                                                                                                                            .toList());

    if (metadataTypes.size() == 0) {
      return TypesHelper.getTypeBuilder().nullType().build();
    } else if (metadataTypes.size() == 1) {
      return metadataTypes.iterator().next();
    } else {
      final UnionTypeBuilder unionTypeBuilder = TypesHelper.getTypeBuilder().unionType();
      metadataTypes.forEach(unionTypeBuilder::of);
      return unionTypeBuilder.build();
    }
  }

}
