package org.mule.datasense.impl.phases.builder;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import org.mule.datasense.api.metadataprovider.ApplicationModel;
import org.mule.datasense.api.metadataprovider.DataSenseProvider;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;

import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Deque;
import java.util.List;

import static java.util.Optional.ofNullable;
import static org.mule.runtime.ast.api.util.MuleAstUtils.parameterOfType;
import static org.mule.runtime.extension.api.stereotype.MuleStereotypes.FLOW;
import static org.mule.runtime.extension.api.stereotype.MuleStereotypes.PROCESSOR;
import static org.mule.runtime.extension.api.stereotype.MuleStereotypes.SUB_FLOW;

public class SimpleDependenciesResolutionScopeStrategy extends DependenciesResolutionScopeStrategy {

  private final List<String> included;

  public SimpleDependenciesResolutionScopeStrategy(ApplicationModel applicationModel,
                                                   DataSenseProvider dataSenseProvider,
                                                   DataSenseResolutionScopeStrategy dataSenseResolutionScopeStrategy) {
    super(applicationModel, dataSenseProvider, dataSenseResolutionScopeStrategy);
    included = initLocations();
  }

  private List<String> initLocations() {
    List<String> result = new ArrayList<>();

    final ArtifactAst muleApplicationModel = getApplicationModel().getMuleApplicationModel();

    Multimap<String, String> flowDependenciesByFlow = ArrayListMultimap.create();

    muleApplicationModel
        .topLevelComponentsStream()
        .filter(topLevelComp -> topLevelComp.getComponentId().isPresent())
        .forEach(topLevelComp -> topLevelComp.recursiveStream()
            .forEach(comp -> {
              topLevelComp.getComponentId().ifPresent(flowName -> {
                parameterOfType(comp, FLOW)
                    .ifPresent(param -> ofNullable(param.getRawValue())
                        .ifPresent(flowRefName -> flowDependenciesByFlow.put(flowName, flowRefName)));
                parameterOfType(comp, SUB_FLOW)
                    .ifPresent(param -> ofNullable(param.getRawValue())
                        .ifPresent(flowRefName -> flowDependenciesByFlow.put(flowName, flowRefName)));

                parameterOfType(comp, PROCESSOR)
                    .ifPresent(param -> muleApplicationModel
                        .filteredComponents(componentModel -> componentModel.getComponentId()
                            .map(name -> param.getRawValue().equals(name))
                            .orElse(false))
                        .forEach(componentModel -> ofNullable(componentModel.getLocation().getRootContainerName())
                            .ifPresent(flowRefName -> flowDependenciesByFlow
                                .put(flowName, flowRefName))));
              });
            }));

    Deque<String> deque = new ArrayDeque<>();

    // prefill queue with accepted flows
    muleApplicationModel
        .topLevelComponentsStream()
        .filter(topLevelComp -> getDataSenseResolutionScopeStrategy().match(topLevelComp))
        .forEach(componentAst -> {
          componentAst.getComponentId().ifPresent(deque::add);
        });

    result.clear();
    while (!deque.isEmpty()) {
      final String flow = deque.pop();
      if (!result.contains(flow)) {
        result.add(flow);
        final Collection<String> flowDependencies = flowDependenciesByFlow.get(flow);
        flowDependencies.forEach(deque::offer);
      }
    }
    return result;
  }

  @Override
  public boolean match(ComponentAst componentModel) {
    return super.match(componentModel) || visited(componentModel);
  }

  private boolean visited(ComponentAst componentModel) {
    return included.contains(componentModel.getComponentId().orElse(null));
  }

}
