package org.mule.datasense.impl.model.ast;

import static java.util.Optional.empty;
import static java.util.Optional.ofNullable;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.component.location.Location;
import org.mule.runtime.ast.api.ComponentAst;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Stream;

import com.google.common.base.Preconditions;

public class MuleApplicationNode extends BaseAstNode {

  private final List<MuleFlowNode> muleFlowNodes;
  private final Map<String, MuleFlowNode> muleFlowNodeByNameMap;
  private final Map<ComponentLocation, MessageProcessorNode> messageProcessorNodeMap;
  private final Map<String, ComponentLocation> componentLocationByNameMap;
  private final Map<String, ComponentLocation> componentLocationByLocationMap;

  public MuleApplicationNode(ComponentIdentifier componentIdentifier, Stream<MuleFlowNode> muleFlowNodes) {
    super(componentIdentifier);
    this.muleFlowNodes = new ArrayList<>();
    this.muleFlowNodeByNameMap = new HashMap<>();
    messageProcessorNodeMap = new HashMap<>();
    componentLocationByNameMap = new HashMap<>();
    componentLocationByLocationMap = new HashMap<>();
    muleFlowNodes.forEach(this::add);
  }

  private void add(MuleFlowNode muleFlowNode) {
    muleFlowNodes.add(muleFlowNode);
    muleFlowNodeByNameMap.put(muleFlowNode.getName(), muleFlowNode);
  }

  public Stream<MuleFlowNode> getMuleFlowNodes() {
    return muleFlowNodes.stream();
  }

  @Override
  public <T> Object accept(AstNodeVisitor<T> astNodeVisitor, T context) {
    return astNodeVisitor.visit(this, context);
  }

  public void register(ComponentLocation componentLocation, MessageProcessorNode messageProcessorNode) {
    Preconditions.checkState(!messageProcessorNodeMap.containsKey(componentLocation),
                             String.format("Node already registered for componentLocation %s", componentLocation));
    messageProcessorNodeMap.put(componentLocation, messageProcessorNode);
    componentLocationByLocationMap.put(componentLocation.getLocation(), componentLocation);
    ofNullable(messageProcessorNode.getComponentModel())
        .flatMap(ComponentAst::getComponentId)
        .ifPresent(name -> componentLocationByNameMap.put(name, componentLocation));
  }

  private Optional<MessageProcessorNode> findMessageProcessorNode(ComponentLocation componentLocation) {
    if (componentLocation == null) {
      return empty();
    }
    return ofNullable(messageProcessorNodeMap.get(componentLocation));
  }

  public Optional<MessageProcessorNode> findMessageProcessorNode(String location) {
    if (location == null) {
      return empty();
    }
    return ofNullable(componentLocationByLocationMap.get(location))
        .flatMap(this::findMessageProcessorNode);
  }

  public Optional<ComponentLocation> findComponentLocationByName(String name) {
    if (name == null) {
      return empty();
    }
    return ofNullable(componentLocationByNameMap.get(name));
  }

  public Optional<MessageProcessorNode> findMessageProcessorNodeByName(String name) {
    if (name == null) {
      return empty();
    }
    return findComponentLocationByName(name)
        .flatMap(this::findMessageProcessorNode);
  }

  public Optional<MessageProcessorNode> findMessageProcessorNode(Location location) {
    return findMessageProcessorNode(location.toString());
  }

  public Optional<MuleFlowNode> findMuleFlowNode(String flow) {
    if (flow == null) {
      return empty();
    }

    return getMuleFlowNodes().filter(muleFlowNode -> flow.equals(muleFlowNode.getName())).findFirst();
  }

  public Stream<MessageProcessorNode> findMessageProcessorNodes(boolean includeRoot) {
    return messageProcessorNodeMap.values().stream()
        .filter(messageProcessorNode -> includeRoot || !messageProcessorNode.isRootMessageProcessorNode());
  }

  public Stream<MessageProcessorNode> findMessageProcessorNodes() {
    return findMessageProcessorNodes(false);
  }

}
