package org.mule.datasense.impl.phases.typing.resolver;

import static java.util.Optional.empty;
import static java.util.Optional.ofNullable;
import static org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.DEFAULT_GROUP_NAME;

import org.mule.datasense.impl.model.annotations.DefinesTypeAnnotation;
import org.mule.datasense.impl.model.annotations.InnerUsesTypeAnnotation;
import org.mule.datasense.impl.model.annotations.MuleApplicationAnnotation;
import org.mule.datasense.impl.model.annotations.TypesResolvedAnnotation;
import org.mule.datasense.impl.model.annotations.UsesTypeAnnotation;
import org.mule.datasense.impl.model.ast.MessageProcessorNode;
import org.mule.datasense.impl.model.ast.MuleApplicationNode;
import org.mule.datasense.impl.model.ast.MuleFlowNode;
import org.mule.datasense.impl.model.types.EventType;
import org.mule.datasense.impl.model.types.TypeUtils;
import org.mule.datasense.impl.phases.builder.ComponentModelType;
import org.mule.datasense.impl.phases.typing.TypingMuleAstVisitor;
import org.mule.datasense.impl.phases.typing.TypingMuleAstVisitorContext;
import org.mule.runtime.api.component.location.ComponentLocation;

import java.util.Optional;

public class AggregatorListenerTypeResolver extends PassThroughTypeResolver {

  public static final String PARAMETER_AGGREGATOR_REF = "aggregatorName";

  private void ensureMuleFlowNodeTypesResolved(MuleFlowNode muleFlowNode, MuleApplicationNode muleApplicationNode,
                                               TypingMuleAstVisitor typingMuleAstVisitor,
                                               TypingMuleAstVisitorContext visitorContext) {
    Optional<TypesResolvedAnnotation> typesResolvedAnnotation =
        muleFlowNode.getAnnotation(TypesResolvedAnnotation.class);
    if (!typesResolvedAnnotation.isPresent()) {
      // attempt from inference
      typingMuleAstVisitor.resolveType(muleFlowNode, TypeUtils.createEventType(null),
                                       visitorContext.getAstNotification(), muleApplicationNode);
    }
  }

  private void ensureComponentLocationTypesResolved(MuleApplicationNode muleApplicationNode, ComponentLocation componentLocation,
                                                    TypingMuleAstVisitor typingMuleAstVisitor,
                                                    TypingMuleAstVisitorContext visitorContext) {
    final String rootContainerName = componentLocation.getRootContainerName();
    muleApplicationNode.findMuleFlowNode(rootContainerName)
        .ifPresent(muleFlowNode -> ensureMuleFlowNodeTypesResolved(muleFlowNode, muleApplicationNode, typingMuleAstVisitor,
                                                                   visitorContext));
  }

  private Optional<MessageProcessorNode> findAggregatorMessageProcessorNode(String aggregatorRef,
                                                                            MuleApplicationNode muleApplicationNode,
                                                                            TypingMuleAstVisitor typingMuleAstVisitor,
                                                                            TypingMuleAstVisitorContext visitorContext) {
    final Optional<MessageProcessorNode> messageProcessorNodeOptional =
        muleApplicationNode.findMessageProcessorNodeByName(aggregatorRef);
    if (messageProcessorNodeOptional.isPresent()) {
      final MessageProcessorNode messageProcessorNode = messageProcessorNodeOptional.get();
      final ComponentLocation componentLocation = messageProcessorNode.getComponentModel().getLocation();
      ensureComponentLocationTypesResolved(muleApplicationNode, componentLocation, typingMuleAstVisitor, visitorContext);

    }
    return messageProcessorNodeOptional;
  }

  private Optional<EventType> resolveAggregatorEventType(String aggregatorRef, MuleApplicationNode muleApplicationNode,
                                                         TypingMuleAstVisitor typingMuleAstVisitor,
                                                         TypingMuleAstVisitorContext visitorContext) {
    if (aggregatorRef == null) {
      return empty();
    }

    return findAggregatorMessageProcessorNode(aggregatorRef, muleApplicationNode, typingMuleAstVisitor, visitorContext)
        .flatMap(aggregatorMessageProcessorNode -> aggregatorMessageProcessorNode.getAnnotation(InnerUsesTypeAnnotation.class))
        .map(InnerUsesTypeAnnotation::getUsesEventType);
  }

  private EventType resolveEventType(MessageProcessorNode messageProcessorNode, TypingMuleAstVisitor typingMuleAstVisitor,
                                     TypingMuleAstVisitorContext visitorContext) {
    MuleApplicationNode muleApplicationNode = visitorContext.getAnnotation(MuleApplicationAnnotation.class)
        .map(MuleApplicationAnnotation::getMuleApplicationNode).orElse(null);
    if (muleApplicationNode == null) {
      return new EventType();
    }

    return ofNullable(messageProcessorNode.getComponentModel())
        .flatMap(componentModel -> resolveAggregatorEventType((String) componentModel
            .getParameter(DEFAULT_GROUP_NAME, PARAMETER_AGGREGATOR_REF)
            .getValue().getRight(),
                                                              muleApplicationNode, typingMuleAstVisitor, visitorContext))
        .orElse(new EventType());
  }

  @Override
  protected EventType resolve(MessageProcessorNode messageProcessorNode, EventType inputEventType,
                              TypingMuleAstVisitor typingMuleAstVisitor,
                              TypingMuleAstVisitorContext visitorContext) {

    messageProcessorNode.annotate(new UsesTypeAnnotation(new EventType()));
    final EventType eventType = resolveEventType(messageProcessorNode, typingMuleAstVisitor, visitorContext);
    messageProcessorNode
        .annotate(new DefinesTypeAnnotation(eventType));

    return eventType;
  }


  @Override
  public Optional<ComponentModelType> getComponentModelType() {
    return Optional.of(ComponentModelType.MESSAGE_SOURCE_NODE);
  }

}
