/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.encryption.jce;

import static java.lang.String.format;
import static javax.crypto.Cipher.DECRYPT_MODE;
import static javax.crypto.Cipher.ENCRYPT_MODE;

import org.mule.encryption.Encrypter;
import org.mule.encryption.exception.MuleEncryptionException;
import org.mule.encryption.exception.MuleInvalidAlgorithmConfigurationException;
import org.mule.encryption.exception.MuleInvalidKeyException;
import org.mule.encryption.key.EncryptionKeyFactory;

import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.SecureRandom;
import java.util.Arrays;

import javax.crypto.Cipher;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.spec.IvParameterSpec;

public class JCEEncrypter implements Encrypter {

  private static final String INSTALL_JCE_MESSAGE = " You need to install the Java Cryptography Extension (JCE) " +
      "Unlimited Strength Jurisdiction Policy Files";

  private static final String ECB = "ECB";

  private final String provider;
  private final String transformation;
  private final EncryptionKeyFactory keyFactory;

  public JCEEncrypter(String transformation, EncryptionKeyFactory keyFactory) {
    this(transformation, null, keyFactory);
  }

  public JCEEncrypter(String transformation, String provider, EncryptionKeyFactory keyFactory) {
    this.transformation = transformation;
    this.provider = provider;
    this.keyFactory = keyFactory;
  }

  @Override
  public byte[] decrypt(byte[] content) throws MuleEncryptionException {
    return runCipher(content, keyFactory.decryptionKey(), DECRYPT_MODE);
  }

  @Override
  public byte[] encrypt(byte[] content) throws MuleEncryptionException {
    return runCipher(content, keyFactory.encryptionKey(), ENCRYPT_MODE);
  }

  protected void initCipher(Cipher cipher, Key cipherKey, int mode)
      throws InvalidKeyException, InvalidAlgorithmParameterException {
    String[] cipherParts = transformation.split("/");

    if (cipherParts.length >= 2 && ECB.equals(cipherParts[1])) {
      cipher.init(mode, cipherKey);
    } else {
      IvParameterSpec ips = new IvParameterSpec(Arrays.copyOfRange(cipherKey.getEncoded(), 0, cipher.getBlockSize()));
      cipher.init(mode, cipherKey, ips, new SecureRandom());
    }
  }

  private byte[] runCipher(byte[] content, Key key, int mode) throws MuleEncryptionException {
    try {

      Cipher cipher = getCipher();
      initCipher(cipher, key, mode);
      return cipher.doFinal(content);

    } catch (InvalidAlgorithmParameterException e) {
      throw invalidAlgorithmConfigurationException(format("Wrong configuration for algorithm '%s'", transformation), e);
    } catch (NoSuchAlgorithmException e) {
      throw invalidAlgorithmConfigurationException(format("Cipher '%s' not found", transformation), e);
    } catch (NoSuchPaddingException e) {
      throw invalidAlgorithmConfigurationException(format("Invalid padding selected for cipher '%s'", transformation), e);
    } catch (NoSuchProviderException e) {
      throw invalidAlgorithmConfigurationException(format("Provider '%s' not found", provider), e);
    } catch (InvalidKeyException e) {
      throw handleInvalidKeyException(e, new String(key.getEncoded()));
    } catch (Exception e) {
      throw new MuleEncryptionException("Could not encrypt or decrypt the data.", e);
    }
  }

  private Cipher getCipher() throws NoSuchPaddingException, NoSuchAlgorithmException, NoSuchProviderException {
    return provider == null ? Cipher.getInstance(transformation) : Cipher.getInstance(transformation, provider);
  }

  private MuleEncryptionException invalidAlgorithmConfigurationException(String message, Exception e) {
    if (!JCE.isJCEInstalled()) {
      message += INSTALL_JCE_MESSAGE;
    }

    return new MuleInvalidAlgorithmConfigurationException(message, e);
  }

  private MuleEncryptionException handleInvalidKeyException(InvalidKeyException e, String key) {
    String message = format("The key is invalid, please make sure it's of a supported size (actual is %s)", key.length());
    return new MuleInvalidKeyException(message, e);
  }
}
