/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.encryption;

import static org.apache.commons.lang.RandomStringUtils.randomAlphabetic;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.junit.Assume.assumeFalse;
import static org.junit.rules.ExpectedException.none;
import static org.mule.encryption.jce.JCE.isJCEInstalled;
import static org.mule.encryption.AllureConstants.EncryptionFeature.ENCRYPTION;
import static org.mule.encryption.AllureConstants.EncryptionFeature.JceEncryptionStory.JCE_ENCRYPTION;
import static java.security.Security.addProvider;
import static java.security.Security.removeProvider;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.mule.encryption.exception.MuleEncryptionException;
import org.mule.encryption.jce.JCEEncrypter;
import org.mule.encryption.key.EncryptionKeyFactory;
import org.mule.encryption.key.SymmetricKeyFactory;
import org.bouncycastle.jce.provider.BouncyCastleProvider;

import java.security.Key;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.NoSuchAlgorithmException;
import java.security.Provider;

import javax.crypto.Cipher;
import javax.crypto.KeyGenerator;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;

import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

@Feature(ENCRYPTION)
@Story(JCE_ENCRYPTION)
public class JCEEncrypterTestCase {

  private static final String PAYLOAD = "Payload to Encrypt";
  private static final String KEY = "posofj00posofj00";
  private static final String ANOTHER_KEY = "differentKey1234";

  private static final String AES = "AES";
  private static final String ARCFOUR = "ARCFOUR";
  private static final String Blowfish = "Blowfish";
  private static final String DES = "DES";
  private static final String DESede = "DESede";
  private static final String RC2 = "RC2";
  private static final String RSA = "RSA";

  // Ciphers
  private static final String AES_CBC_PKCS5PADDING_CIPHER = "AES/CBC/PKCS5Padding";
  private static final String AES_ECB_PKCS5PADDING_CIPHER = "AES/ECB/PKCS5Padding";
  private static final String ARCFOUR_ECB_NOPADDING_CIPHER = "ARCFOUR/ECB/NoPadding";
  private static final String BLOWFISH_CBC_PKCS5PADDING_CIPHER = "Blowfish/CBC/PKCS5Padding";
  private static final String DES_CBC_PKCS5PADDING_CIPHER = "DES/CBC/PKCS5Padding";
  private static final String DESEDE_CBC_PKCS5PADDING_CIPHER = "DESede/CBC/PKCS5Padding";
  private static final String RC2_CBC_PKCS5PADDING_CIPHER = "RC2/CBC/PKCS5Padding";
  private static final String RSA_ECB_OAEPWithSHA_256AndMGF1PaddingCIPHER = "RSA/ECB/OAEPWithSHA-256AndMGF1Padding";
  private static final String RSA_NONE_OAEPWithSHA_1AndMGF1Padding = "RSA/NONE/OAEPWithSHA1AndMGF1Padding";

  @Rule
  public ExpectedException expectedException = none();

  private Provider provider;

  private JCEEncrypter encrypter;

  @Before
  public void setup() {
    EncryptionKeyFactory keyFactory = keyFactory(KEY);

    encrypter = new JCEEncrypter(AES_CBC_PKCS5PADDING_CIPHER, keyFactory, false);

    provider = new BouncyCastleProvider();

    addProvider(provider);
  }

  @After
  public void after() {
    if (provider != null) {
      removeProvider(provider.getName());
    }
  }

  @Description("Encrypt and decrypt payload using AES/CBC/PKCS5Padding")
  @Test
  public void encryptAndDecryptUsingAES_CBC_PKCS5Padding() throws MuleEncryptionException, NoSuchAlgorithmException {
    assertEncryptAndDecryptPayload(AES_CBC_PKCS5PADDING_CIPHER, symmetricKeyFactory(AES));
  }

  @Description("Encrypt and decrypt payload using ECB encryption mode")
  @Test
  public void encryptAndDecryptUsingECBEncryptionMode() throws MuleEncryptionException, NoSuchAlgorithmException {
    assertEncryptAndDecryptPayload(AES_ECB_PKCS5PADDING_CIPHER, symmetricKeyFactory(AES));
  }

  @Description("Encrypt and decrypt payload using ARCFOUR/EBC/NoPadding")
  @Test
  public void encryptAndDecryptUsingARCFOUR_ECB_PKCS5Padding() throws MuleEncryptionException, NoSuchAlgorithmException {
    assertEncryptAndDecryptPayload(ARCFOUR_ECB_NOPADDING_CIPHER, symmetricKeyFactory(ARCFOUR));
  }

  @Description("Encrypt and decrypt payload using Blowfish/CBC/PKCS5Padding")
  @Test
  public void encryptAndDecryptUsingBlowfish_CBC_PKCS5Padding() throws MuleEncryptionException, NoSuchAlgorithmException {
    assertEncryptAndDecryptPayload(BLOWFISH_CBC_PKCS5PADDING_CIPHER, symmetricKeyFactory(Blowfish));
  }

  @Description("Encrypt and decrypt payload using DES/CBC/PKCS5Padding")
  @Test
  public void encryptAndDecryptUsingDES_CBC_PKCS5Padding() throws MuleEncryptionException, NoSuchAlgorithmException {
    assertEncryptAndDecryptPayload(DES_CBC_PKCS5PADDING_CIPHER, symmetricKeyFactory(DES));
  }

  @Description("Encrypt and decrypt payload using DESede/CBC/PKCS5Padding")
  @Test
  public void encryptAndDecryptUsingDESed_CBC_PKCS5Padding() throws MuleEncryptionException, NoSuchAlgorithmException {
    assertEncryptAndDecryptPayload(DESEDE_CBC_PKCS5PADDING_CIPHER, symmetricKeyFactory(DESede));
  }

  @Description("Encrypt and decrypt payload using RC2/CBC/PKCS5Padding")
  @Test
  public void encryptAndDecryptUsingRC2_CBC_PKCS5Padding() throws MuleEncryptionException, NoSuchAlgorithmException {
    assertEncryptAndDecryptPayload(RC2_CBC_PKCS5PADDING_CIPHER, symmetricKeyFactory(RC2));
  }

  @Description("Encrypt and decrypt payload using RSA/ECB/OAEPWithSHA-256AndMGF1Padding")
  @Test
  public void encryptAndDecryptUsingRSA_ECB_OAEPWithSHA_256AndMGF1Padding()
      throws MuleEncryptionException, NoSuchAlgorithmException {
    assertEncryptAndDecryptPayload(RSA_ECB_OAEPWithSHA_256AndMGF1PaddingCIPHER, asymmetricKeyFactory(RSA));
  }

  @Description("Encrypt and decrypt payload using RSA/NONE/OAEPWithSHA1AndMGF1Padding")
  @Test
  public void encryptAndDecryptUsingRSA_NONE_OAEPWithSHA_1AndMGF1Padding()
      throws MuleEncryptionException, NoSuchAlgorithmException {
    assertEncryptAndDecryptPayload(RSA_NONE_OAEPWithSHA_1AndMGF1Padding, asymmetricKeyFactory(RSA));
  }

  private void assertEncryptAndDecryptPayload(String transformation, EncryptionKeyFactory keyFactory)
      throws MuleEncryptionException {
    // Assert encrypt and decrypt payload
    assertEncryptAndDecryptPayload(new JCEEncrypter(transformation, keyFactory, false));

    // Assert encrypt and decrypt payload with using random IV
    assertEncryptAndDecryptPayload(new JCEEncrypter(transformation, keyFactory, true));
  }

  private void assertEncryptAndDecryptPayload(JCEEncrypter encrypter) throws MuleEncryptionException {
    byte[] content = PAYLOAD.getBytes();

    byte[] encrypted = encrypter.encrypt(content);
    byte[] decrypted = encrypter.decrypt(encrypted);

    assertThat(new String(decrypted), is(PAYLOAD));
  }

  @Test
  public void errorEncrypting() throws Exception {
    JCEEncrypter anotherEncrypter = new JCEEncrypter(AES_CBC_PKCS5PADDING_CIPHER, keyFactory(ANOTHER_KEY));
    byte[] encrypted = encrypter.encrypt(PAYLOAD.getBytes());

    expectedException.expect(MuleEncryptionException.class);
    expectedException.expectMessage("Could not encrypt or decrypt the data.");

    anotherEncrypter.decrypt(encrypted);
  }

  @Test
  public void invalidAlgorithm() throws Exception {
    JCEEncrypter anotherEncrypter = new JCEEncrypter("invalid/CBC/PKCS5Padding", keyFactory(ANOTHER_KEY));
    byte[] encrypted = encrypter.encrypt(PAYLOAD.getBytes());

    expectedException.expect(MuleEncryptionException.class);
    expectedException.expectMessage("Cipher 'invalid/CBC/PKCS5Padding' not found");

    anotherEncrypter.decrypt(encrypted);
  }

  @Test
  public void invalidMode() throws Exception {
    JCEEncrypter anotherEncrypter = new JCEEncrypter("AES/invalid/PKCS5Padding", keyFactory(ANOTHER_KEY));
    byte[] encrypted = encrypter.encrypt(PAYLOAD.getBytes());

    expectedException.expect(MuleEncryptionException.class);
    expectedException.expectMessage("Cipher 'AES/invalid/PKCS5Padding' not found");

    anotherEncrypter.decrypt(encrypted);
  }

  @Test
  public void invalidPadding() throws Exception {
    JCEEncrypter anotherEncrypter = new JCEEncrypter("AES/CBC/invalid", keyFactory(ANOTHER_KEY));
    byte[] encrypted = encrypter.encrypt(PAYLOAD.getBytes());

    expectedException.expect(MuleEncryptionException.class);
    expectedException.expectMessage("Cipher 'AES/CBC/invalid' not found");

    anotherEncrypter.decrypt(encrypted);
  }

  @Test
  public void invalidProvider() throws Exception {
    JCEEncrypter anotherEncrypter = new JCEEncrypter(AES_CBC_PKCS5PADDING_CIPHER, "invalid", keyFactory(ANOTHER_KEY));
    byte[] encrypted = encrypter.encrypt(PAYLOAD.getBytes());

    expectedException.expect(MuleEncryptionException.class);
    expectedException.expectMessage("Provider 'invalid' not found");

    anotherEncrypter.decrypt(encrypted);
  }

  @Test
  public void shortKey() throws Exception {
    JCEEncrypter anotherEncrypter = new JCEEncrypter(AES_CBC_PKCS5PADDING_CIPHER, keyFactory("shortKey"));

    expectedException.expect(MuleEncryptionException.class);
    expectedException.expectMessage("The key is invalid, please make sure it's of a supported size (actual is 8)");

    anotherEncrypter.encrypt(PAYLOAD.getBytes());
  }

  @Test
  public void longKey() throws Exception {
    assumeFalse(isJCEInstalled());
    int maxKeySize = Cipher.getMaxAllowedKeyLength(AES) / 8;
    JCEEncrypter anotherEncrypter = new JCEEncrypter(AES_CBC_PKCS5PADDING_CIPHER, keyFactory(randomAlphabetic(maxKeySize + 1)));

    expectedException.expect(MuleEncryptionException.class);
    expectedException
        .expectMessage("The key is invalid, please make sure it's of a supported size (actual is " + (maxKeySize + 1) + ")");

    anotherEncrypter.encrypt(PAYLOAD.getBytes());
  }

  private EncryptionKeyFactory keyFactory(String key) {
    return symmetricKeyFactory(new SecretKeySpec(key.getBytes(), AES));
  }

  private EncryptionKeyFactory symmetricKeyFactory(String algorithm) throws NoSuchAlgorithmException {
    KeyGenerator keygenerator = KeyGenerator.getInstance(algorithm);
    return symmetricKeyFactory(keygenerator.generateKey());
  }

  private EncryptionKeyFactory symmetricKeyFactory(SecretKey key) {
    return (SymmetricKeyFactory) () -> key;
  }

  private EncryptionKeyFactory asymmetricKeyFactory(String algorithm) throws NoSuchAlgorithmException {
    KeyPairGenerator keyPairGenerator = KeyPairGenerator.getInstance(algorithm);
    KeyPair keyPair = keyPairGenerator.generateKeyPair();

    return new EncryptionKeyFactory() {

      @Override
      public Key encryptionKey() {
        return keyPair.getPublic();
      }

      @Override
      public Key decryptionKey() {
        return keyPair.getPrivate();
      }
    };
  }
}
