/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.encryption.jce;

import static java.lang.String.format;
import static java.lang.System.arraycopy;
import static javax.crypto.Cipher.DECRYPT_MODE;
import static javax.crypto.Cipher.ENCRYPT_MODE;

import org.mule.encryption.Encrypter;
import org.mule.encryption.exception.MuleEncryptionException;
import org.mule.encryption.exception.MuleInvalidAlgorithmConfigurationException;
import org.mule.encryption.exception.MuleInvalidKeyException;
import org.mule.encryption.key.EncryptionKeyFactory;

import java.security.Provider;
import java.security.Key;
import java.security.SecureRandom;
import java.security.InvalidAlgorithmParameterException;
import java.security.NoSuchProviderException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.spec.AlgorithmParameterSpec;
import java.util.Arrays;

import javax.crypto.Cipher;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.spec.IvParameterSpec;

public class JCEEncrypter implements Encrypter {

  private static final String INSTALL_JCE_MESSAGE = " You need to install the Java Cryptography Extension (JCE) " +
      "Unlimited Strength Jurisdiction Policy Files";

  private static final String ECB = "ECB";
  private static final String RSA = "RSA";
  private static final String NONE = "NONE";

  private final String provider;
  private final String transformation;
  private final EncryptionKeyFactory keyFactory;
  private final boolean useRandomIV;
  private final Provider providerInstance;

  /**
   * Initializes this class with a specific key factory and security provider instance, setting the password salt does not have to
   * be generated using a random IV. The transformations created with this encryptor make use of the best suitable security
   * provider from the Java security providers list to initialize the javax.crypto.Cipher.
   * 
   * @param transformation The algorithm used for transformation.
   * @param keyFactory     The key factory used for transformation.
   */
  public JCEEncrypter(String transformation, EncryptionKeyFactory keyFactory) {
    this(transformation, null, keyFactory);
  }

  /**
   * Initializes this class with a specific key factory and security provider instance, setting the password salt does not have to
   * be generated using a random IV. The transformations created with this encryptor make use of the security provider name set as
   * parameter to initialize the javax.crypto.Cipher, looking for it on the list of Java security providers configured on the JVM.
   * 
   * @param transformation The algorithm used for transformation.
   * @param provider       The security provider name to use for this transformation.
   * @param keyFactory     The key factory used for transformation.
   */
  public JCEEncrypter(String transformation, String provider, EncryptionKeyFactory keyFactory) {
    this(transformation, provider, keyFactory, false);
  }

  /**
   * Initializes this class with a specific key factory and security provider instance, allowing to determine if the password salt
   * has to be generated using a random IV. The transformations created with this encryptor make use of the best suitable security
   * provider from the Java security providers list to initialize the javax.crypto.Cipher.
   * 
   * @param transformation The algorithm used for transformation.
   * @param keyFactory     The key factory used for transformation.
   * @param useRandomIV    A flag to determine if the password salt must be generated using a random IV.
   */
  public JCEEncrypter(String transformation, EncryptionKeyFactory keyFactory, boolean useRandomIV) {
    this(transformation, null, keyFactory, useRandomIV);
  }

  /**
   * Initializes this class with a specific key factory and security provider instance, setting the password salt does not have to
   * be generated using a random IV. The transformations created with this encryptor make use of the security provider instance
   * set as parameter to initialize the javax.crypto.Cipher instead of looking for it on the list of Java security providers
   * configured on the JVM.
   * 
   * @param transformation The algorithm used for transformation.
   * @param keyFactory     The key factory used for transformation.
   * @param provider       The security provider used for transformation.
   */
  public JCEEncrypter(String transformation, EncryptionKeyFactory keyFactory, Provider provider) {
    this(transformation, keyFactory, false, provider);
  }

  /**
   * Initializes this class with a specific key factory and security provider instance, allowing to determine if the password salt
   * has to be generated using a random IV. The transformations created with this encryptor make use of the security provider
   * instance set as parameter to initialize the javax.crypto.Cipher instead of looking for it on the list of Java security
   * providers configured on the JVM.
   * 
   * @param transformation The algorithm used for transformation.
   * @param keyFactory     The key factory used for transformation.
   * @param useRandomIV    A flag to determine if the password salt must be generated using a random IV.
   * @param provider       The security provider used for transformation.
   */
  public JCEEncrypter(String transformation, EncryptionKeyFactory keyFactory, boolean useRandomIV, Provider provider) {
    this.transformation = transformation;
    this.keyFactory = keyFactory;
    this.useRandomIV = useRandomIV;
    this.providerInstance = provider;
    this.provider = null;
  }

  /**
   * Initializes this class with a specific key factory and security provider instance, allowing to determine if the password salt
   * has to be generated using a random IV. The transformations created with this encryptor make use of the security provider name
   * set as parameter to initialize the javax.crypto.Cipher, looking for it on the list of Java security providers configured on
   * the JVM.
   * 
   * @param transformation The algorithm used for transformation.
   * @param provider       The security provider name to use for this transformation.
   * @param keyFactory     The key factory used for transformation.
   * @param useRandomIV    A flag to determine if the password salt must be generated using a random IV.
   */
  public JCEEncrypter(String transformation, String provider, EncryptionKeyFactory keyFactory, boolean useRandomIV) {
    this.transformation = transformation;
    this.provider = provider;
    this.keyFactory = keyFactory;
    this.useRandomIV = useRandomIV;
    this.providerInstance = null;
  }

  @Override
  public byte[] decrypt(byte[] content) throws MuleEncryptionException {
    return runCipher(content, keyFactory.decryptionKey(), DECRYPT_MODE);
  }

  @Override
  public byte[] encrypt(byte[] content) throws MuleEncryptionException {
    return runCipher(content, keyFactory.encryptionKey(), ENCRYPT_MODE);
  }

  @Override
  public String getTransformation() {
    return transformation;
  }

  protected AlgorithmParameterSpec getAlgorithmParameterSpec(IvParameterSpec ivParam) {
    return ivParam;
  }

  private boolean doesNotUseIV() {
    // ECB Encryption mode does not use IV
    // RSA/NONE Encryption does not use IV: https://www.mulesoft.org/jira/browse/CRYPT-22#
    String[] cipherParts = transformation.split("/");
    return cipherParts.length >= 2 &&
        (ECB.equals(cipherParts[1]) || RSA.equals(cipherParts[0]) && NONE.equalsIgnoreCase(cipherParts[1]));
  }

  private byte[] runCipher(byte[] content, Key key, int mode) throws MuleEncryptionException {
    try {

      Cipher cipher = getCipher();

      if (doesNotUseIV()) {
        cipher.init(mode, key);

        return cipher.doFinal(content);
      } else {
        SecureRandom secureRandom = new SecureRandom();

        // Create IV
        byte[] ivInByteArray = new byte[cipher.getBlockSize()];;
        if (useRandomIV) {
          if (mode == ENCRYPT_MODE) {
            secureRandom.nextBytes(ivInByteArray);
          } else {
            ivInByteArray = Arrays.copyOfRange(content, 0, ivInByteArray.length);
            content = Arrays.copyOfRange(content, ivInByteArray.length, content.length);
          }
        } else {
          ivInByteArray = Arrays.copyOfRange(key.getEncoded(), 0, ivInByteArray.length);
        }

        cipher.init(mode, key, getAlgorithmParameterSpec(new IvParameterSpec(ivInByteArray)), secureRandom);

        byte[] result = cipher.doFinal(content);

        if (mode == ENCRYPT_MODE && useRandomIV) {
          byte[] byteArrayToConcatEncryptedDataAndIV = new byte[ivInByteArray.length + result.length];

          arraycopy(ivInByteArray, 0, byteArrayToConcatEncryptedDataAndIV, 0, ivInByteArray.length);
          arraycopy(result, 0, byteArrayToConcatEncryptedDataAndIV, ivInByteArray.length, result.length);

          return byteArrayToConcatEncryptedDataAndIV;
        }

        return result;
      }

    } catch (InvalidAlgorithmParameterException e) {
      throw invalidAlgorithmConfigurationException(format("Wrong configuration for algorithm '%s'", transformation), e);
    } catch (NoSuchAlgorithmException e) {
      throw invalidAlgorithmConfigurationException(format("Cipher '%s' not found", transformation), e);
    } catch (NoSuchPaddingException e) {
      throw invalidAlgorithmConfigurationException(format("Invalid padding selected for cipher '%s'", transformation), e);
    } catch (NoSuchProviderException e) {
      throw invalidAlgorithmConfigurationException(format("Provider '%s' not found", provider), e);
    } catch (InvalidKeyException e) {
      throw handleInvalidKeyException(e, new String(key.getEncoded()));
    } catch (Exception e) {
      throw new MuleEncryptionException("Could not encrypt or decrypt the data.", e);
    }
  }

  /**
   * Initializes a javax.crypto.Cipher using the security provider name or instance set on this class constructor. If none is
   * provided, the Cipher chooses the best suitable security provider to run on the transformation.
   * 
   * @return javax.crypto.Cipher initialized with the proper parameters to run the transformation.
   * @throws NoSuchPaddingException   This exception is thrown when a particular padding mechanism is requested but is not
   *                                  available in the environment.
   * @throws NoSuchAlgorithmException This exception is thrown when a particular cryptographic algorithm is requested but is not
   *                                  available in the environment.
   * @throws NoSuchProviderException  This exception is thrown when a particular security provider is requested but is not
   *                                  available in the environment.
   */
  private Cipher getCipher() throws NoSuchPaddingException, NoSuchAlgorithmException, NoSuchProviderException {
    Cipher cipher;
    if (provider != null) {
      cipher = Cipher.getInstance(transformation, provider);
    } else if (providerInstance != null) {
      cipher = Cipher.getInstance(transformation, providerInstance);
    } else {
      cipher = Cipher.getInstance(transformation);
    }
    return cipher;
  }

  private MuleEncryptionException invalidAlgorithmConfigurationException(String message, Exception e) {
    if (!JCE.isJCEInstalled()) {
      message += INSTALL_JCE_MESSAGE;
    }

    return new MuleInvalidAlgorithmConfigurationException(message, e);
  }

  private MuleEncryptionException handleInvalidKeyException(InvalidKeyException e, String key) {
    String message = format("The key is invalid, please make sure it's of a supported size (actual is %s)", key.length());
    return new MuleInvalidKeyException(message, e);
  }
}
