/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.encryption;

import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.junit.Assert.assertThrows;
import static org.junit.Assert.fail;
import static org.junit.rules.ExpectedException.none;

import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.junit.After;
import org.mule.encryption.exception.MuleEncryptionException;
import org.mule.encryption.jce.JCEPbeEncrypter;
import org.mule.encryption.key.EncryptionKeyFactory;
import org.mule.encryption.key.SymmetricKeyFactory;

import java.security.NoSuchAlgorithmException;
import java.security.Provider;
import java.security.Security;
import java.security.spec.InvalidKeySpecException;
import java.util.Base64;

import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.PBEKeySpec;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

public class JCEPbeEncrypterTestCase {

  private static final String PAYLOAD = "Payload to Encrypt";
  private static final String PASSWORD = "a secure password";
  private static final String BASE64_ENCRYPTED_PAYLOAD = "9EdOtFMo1Gg5/BFg7iLwnN26+nO+JA8Kril0rq8hPhg=";

  private EncryptionKeyFactory keyFactory;

  private static final String TRANSFORMATION = "PBEWithHmacSHA256AndAES_128";

  private static final Provider provider = new BouncyCastleProvider();

  @Before
  public void setup() {
    keyFactory = keyFactory(PASSWORD);
  }

  @After
  public void shutDown() {
    Security.removeProvider(provider.getName());
  }

  @Test
  public void encryptByteArrayDefaultProvider() throws MuleEncryptionException {
    JCEPbeEncrypter encryptor = new JCEPbeEncrypter(TRANSFORMATION, keyFactory);
    byte[] content = PAYLOAD.getBytes();

    byte[] encrypted = encryptor.encrypt(content);

    String base64Encrypted = new String(Base64.getEncoder().encode(encrypted));
    assertThat(base64Encrypted, is(BASE64_ENCRYPTED_PAYLOAD));
  }

  @Test
  public void encryptByteArrayBCInstance() {
    // Bouncy Castle does not support PBEWithHmacSHA256AndAES_128
    JCEPbeEncrypter encryptor = new JCEPbeEncrypter(TRANSFORMATION, keyFactory, provider);

    byte[] content = PAYLOAD.getBytes();

    assertThrows(org.mule.encryption.exception.MuleInvalidAlgorithmConfigurationException.class,
                 () -> encryptor.encrypt(content));

  }

  @Test
  public void encryptByteArrayBCName() {
    // Bouncy Castle does not support PBEWithHmacSHA256AndAES_128
    Security.addProvider(provider);
    JCEPbeEncrypter encryptor = new JCEPbeEncrypter(TRANSFORMATION, provider.getName(), keyFactory);

    byte[] content = PAYLOAD.getBytes();

    assertThrows(org.mule.encryption.exception.MuleInvalidAlgorithmConfigurationException.class,
                 () -> encryptor.encrypt(content));
  }

  private EncryptionKeyFactory keyFactory(String password) {
    return (SymmetricKeyFactory) () -> {
      try {
        PBEKeySpec keySpec = new PBEKeySpec(password.toCharArray());
        SecretKeyFactory keyFactory = SecretKeyFactory.getInstance("PBE");
        return keyFactory.generateSecret(keySpec);
      } catch (InvalidKeySpecException | NoSuchAlgorithmException e) {
        throw new RuntimeException(e);
      }
    };
  }
}
