/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.encryption.jce;

import org.mule.encryption.key.EncryptionKeyFactory;

import java.security.Provider;
import java.security.spec.AlgorithmParameterSpec;

import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.PBEParameterSpec;

public class JCEPbeEncrypter extends JCEEncrypter {

  /**
   * Initializes this class with a specific key factory and security provider instance. The transformations created with this
   * encryptor searches for the best suitable security provider in the list of Java security providers configured on the JVM to
   * initialize the javax.crypto.Cipher
   * 
   * @param transformation The algorithm used for transformation.
   * @param keyFactory     The key factory used for transformation.
   */
  public JCEPbeEncrypter(String transformation, EncryptionKeyFactory keyFactory) {
    this(transformation, null, keyFactory);
  }

  /**
   * Initializes this class with a specific key factory and security provider instance. The transformations created with this
   * encryptor make use of the security provider name set as parameter to initialize the javax.crypto.Cipher looking for it on the
   * list of Java security providers configured on the JVM.
   *
   * @param transformation The algorithm used for transformation.
   * @param keyFactory     The key factory used for transformation.
   * @param provider       The name of the security provider used for transformation.
   */
  public JCEPbeEncrypter(String transformation, String provider, EncryptionKeyFactory keyFactory) {
    super(transformation, provider, keyFactory);
  }

  /**
   * Initializes this class with a specific key factory and security provider instance. The transformations created with this
   * encryptor make use of the security provider instance set as parameter to initialize the javax.crypto.Cipher instead of
   * looking for it on the list of Java security providers configured on the JVM.
   * 
   * @param transformation The algorithm used for transformation.
   * @param keyFactory     The key factory used for transformation.
   * @param provider       The security provider used for transformation.
   */
  public JCEPbeEncrypter(String transformation, EncryptionKeyFactory keyFactory, Provider provider) {
    super(transformation, keyFactory, provider);
  }

  @Override
  protected AlgorithmParameterSpec getAlgorithmParameterSpec(IvParameterSpec ivParam) {
    return new PBEParameterSpec("12345678".getBytes(), 20, ivParam);
  }
}
