/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.adapter1x;

import static org.mule.maven.client.adapter1x.BundleUtils.dependenciesTo1x;
import static org.mule.maven.client.adapter1x.BundleUtils.dependencyTo1x;
import static org.mule.maven.client.adapter1x.BundleUtils.descriptorFrom1x;
import static org.mule.maven.client.adapter1x.BundleUtils.descriptorsFrom1x;

import org.mule.maven.client.api.LocalRepositorySupplierFactory;
import org.mule.maven.client.api.MavenClient;
import org.mule.maven.client.api.MavenClientProvider;
import org.mule.maven.client.api.MavenReactorResolver;
import org.mule.maven.client.api.PomFileSupplierFactory;
import org.mule.maven.client.api.SettingsSupplierFactory;
import org.mule.maven.client.api.VersionRangeResult;
import org.mule.maven.client.api.model.BundleDependency;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.client.internal.MuleMavenClient;
import org.mule.maven.client.internal.MuleMavenClientProvider;

import java.io.File;
import java.util.List;
import java.util.Optional;

import org.apache.maven.model.Model;

/**
 * Implementation of the {@link MavenClientProvider} 1.x that delegates to the 2.x implementation.
 */
public class MavenClientProviderAdapter implements MavenClientProvider {

  private final MuleMavenClientProvider adaptedProvider = new MuleMavenClientProvider();

  @Override
  public MavenClient createMavenClient(MavenConfiguration mavenConfiguration) {
    final MuleMavenClient mavenClient = (MuleMavenClient) adaptedProvider.createMavenClient(mavenConfiguration);
    return new MavenClient() {

      @Override
      public VersionRangeResult resolveVersionRange(BundleDescriptor bundleDescriptor) {
        return mavenClient.resolveVersionRange(descriptorFrom1x(bundleDescriptor));
      }

      @Override
      public List<BundleDependency> resolvePluginBundleDescriptorsDependencies(List<BundleDescriptor> bundleDescriptors) {
        return dependenciesTo1x(mavenClient.resolvePluginBundleDescriptorsDependencies(descriptorsFrom1x(bundleDescriptors)));
      }

      @Override
      public List<BundleDependency> resolveBundleDescriptorDependencies(boolean includeTestDependencies,
                                                                        boolean includeProvidedDependencies,
                                                                        BundleDescriptor bundleDescriptor) {
        return dependenciesTo1x(mavenClient.resolveBundleDescriptorDependencies(includeTestDependencies,
                                                                                includeProvidedDependencies,
                                                                                descriptorFrom1x(bundleDescriptor)));
      }

      @Override
      public List<BundleDependency> resolveBundleDescriptorDependencies(boolean includeTestDependencies,
                                                                        BundleDescriptor bundleDescriptor) {
        return dependenciesTo1x(mavenClient.resolveBundleDescriptorDependencies(includeTestDependencies,
                                                                                descriptorFrom1x(bundleDescriptor)));
      }

      @Override
      public BundleDependency resolveBundleDescriptor(BundleDescriptor bundleDescriptor) {
        return dependencyTo1x(mavenClient.resolveBundleDescriptor(descriptorFrom1x(bundleDescriptor)));
      }

      @Override
      public List<BundleDependency> resolveArtifactDependencies(File artifactFile, boolean includeTestDependencies,
                                                                boolean includeProvidedDependencies,
                                                                Optional<File> localRepositoryLocationSupplier,
                                                                Optional<MavenReactorResolver> mavenReactorResolver,
                                                                Optional<File> temporaryFolder) {
        return dependenciesTo1x(mavenClient
            .resolveArtifactDependencies(artifactFile, includeTestDependencies, includeProvidedDependencies,
                                         localRepositoryLocationSupplier, mavenReactorResolver, temporaryFolder));
      }

      @Override
      public List<BundleDependency> resolveArtifactDependencies(File artifactFile, boolean includeTestDependencies,
                                                                boolean includeProvidedDependencies,
                                                                Optional<File> localRepositoryLocationSupplier,
                                                                Optional<File> temporaryFolder) {
        return dependenciesTo1x(mavenClient.resolveArtifactDependencies(artifactFile, includeTestDependencies,
                                                                        includeProvidedDependencies,
                                                                        localRepositoryLocationSupplier, temporaryFolder));
      }

      @Override
      public List<BundleDependency> resolveArtifactDependencies(List<BundleDescriptor> dependencies,
                                                                Optional<File> localRepositoryLocationSupplier,
                                                                Optional<MavenReactorResolver> mavenReactorResolver) {
        return dependenciesTo1x(mavenClient.resolveArtifactDependencies(descriptorsFrom1x(dependencies),
                                                                        localRepositoryLocationSupplier, mavenReactorResolver));
      }

      @Override
      public Model getRawPomModel(File artifactFile) {
        // No-op
        return null;
      }

      @Override
      public MavenConfiguration getMavenConfiguration() {
        return mavenClient.getMavenConfiguration();
      }

      @Override
      public Model getEffectiveModel(File artifactFile, Optional<File> temporaryFolder) {
        // No-op
        return null;
      }
    };
  }

  @Override
  public PomFileSupplierFactory getPomFileSuppliers() {
    return adaptedProvider.getPomFileSuppliers();
  }

  @Override
  public LocalRepositorySupplierFactory getLocalRepositorySuppliers() {
    return adaptedProvider.getLocalRepositorySuppliers();
  }

  @Override
  public SettingsSupplierFactory getSettingsSupplierFactory() {
    return adaptedProvider.getSettingsSupplierFactory();
  }

}
