/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.api.model;

import static java.util.Optional.ofNullable;
import static org.mule.maven.client.api.util.Preconditions.checkNotNull;

import java.net.URL;
import java.util.Optional;

/**
 * Immutable implementation of {@link RemoteRepository}
 * 
 * @since 1.0.0
 */
class ImmutableRemoteRepository implements RemoteRepository {

  private String id;
  private URL url;
  private Authentication authentication;

  /**
   * @param id repository id
   * @param url repository URL
   * @param authentication repository authentication credentials.
   */
  public ImmutableRemoteRepository(String id, URL url, Optional<Authentication> authentication) {
    checkNotNull(id, "id cannot be null");
    checkNotNull(url, "url cannot be null");
    this.id = id;
    this.url = url;
    this.authentication = authentication.orElse(null);
  }

  @Override
  public String getId() {
    return id;
  }

  @Override
  public URL getUrl() {
    return url;
  }

  @Override
  public Optional<Authentication> getAuthentication() {
    return ofNullable(authentication);
  }

  @Override
  public String toString() {
    return "ImmutableRemoteRepository{" +
        "id='" + id + '\'' +
        ", url=" + url +
        ", authentication=" + authentication +
        '}';
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    ImmutableRemoteRepository that = (ImmutableRemoteRepository) o;

    if (!id.equals(that.id)) {
      return false;
    }
    if (!url.equals(that.url)) {
      return false;
    }
    return authentication != null ? authentication.equals(that.authentication) : that.authentication == null;
  }

  @Override
  public int hashCode() {
    int result = id.hashCode();
    result = 31 * result + url.hashCode();
    result = 31 * result + (authentication != null ? authentication.hashCode() : 0);
    return result;
  }
}
