/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.maven.client.api.model;

import static java.util.Objects.requireNonNull;
import static java.util.Optional.ofNullable;

import org.mule.maven.client.api.MavenClient;

import java.io.File;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;

/**
 * Configuration settings for a {@link MavenClient}.
 *
 * @since 1.0
 */
public interface MavenConfiguration {

  /**
   * @return {@link File} with the location of the local maven repository.
   */
  File getLocalMavenRepositoryLocation();

  /**
   * @return a list of remote repository to lookup for artifacts. May be empty.
   */
  List<RemoteRepository> getMavenRemoteRepositories();

  /**
   * @return the user settings.xml file location.
   */
  Optional<File> getUserSettingsLocation();

  /**
   *
   * @return the global settings.xml file location.
   */
  Optional<File> getGlobalSettingsLocation();

  /**
   * @return true if for all the repositories, the update policy for artifact should be to never update.
   */
  boolean getForcePolicyUpdateNever();

  /**
   * @return a {@link MavenConfigurationBuilder} to build a {@link MavenConfiguration}.
   */
  static MavenConfigurationBuilder newMavenConfigurationBuilder() {
    return new MavenConfigurationBuilder();
  }

  class MavenConfigurationBuilder {

    private File localMavenRepository;
    private LinkedList<RemoteRepository> remoteRepositories = new LinkedList<>();
    private File userSettingsFile;
    private File globalSettingsFile;
    private boolean forcePolicyUpdateNever;

    /**
     * @param localMavenRepository the local maven repository location.
     * @return this
     */
    public MavenConfigurationBuilder withLocalMavenRepositoryLocation(File localMavenRepository) {
      requireNonNull(localMavenRepository, "localMavenRepository cannot be null");
      this.localMavenRepository = localMavenRepository;
      return this;
    }

    /**
     * Adds a new remote repository to use for discovering artifacts. The order in which the remote repositories are added it's
     * going to be the same order in which they will be used to discover artifacts.
     *
     * @param remoteRepository a remote maven repository
     * @return this
     */
    public MavenConfigurationBuilder withRemoteRepository(RemoteRepository remoteRepository) {
      requireNonNull(remoteRepository, "remoteRepository cannot be null");
      this.remoteRepositories.addLast(remoteRepository);
      return this;
    }

    /**
     * @param userSettingsFile the user settings.xml file, usually located in USER_HOME/.m2/settings.xml
     * @return this
     */
    public MavenConfigurationBuilder withUserSettingsLocation(File userSettingsFile) {
      requireNonNull(userSettingsFile, "userSettingsFile cannot be null");
      this.userSettingsFile = userSettingsFile;
      return this;
    }

    /**
     * @param globalSettingsFile the user settings.xml file, usually located in MAVEN_HOME/conf/settings.xml
     * @return this
     */
    public MavenConfigurationBuilder withGlobalSettingsLocation(File globalSettingsFile) {
      requireNonNull(globalSettingsFile, "globalSettingsFile cannot be null");
      this.globalSettingsFile = globalSettingsFile;
      return this;
    }

    /**
     * @param forcePolicyUpdateNever if true, regardless of the configuration in the settings.xml for the repositories, the update
     *        policies for artifacts will be to never update them
     * @return this
     */
    public MavenConfigurationBuilder withForcePolicyUpdateNever(boolean forcePolicyUpdateNever) {
      requireNonNull(forcePolicyUpdateNever, "forcePolicyUpdateNever cannot be null");
      this.forcePolicyUpdateNever = forcePolicyUpdateNever;
      return this;
    }


    /**
     * Builds the {@link MavenConfiguration} object.
     *
     * @return {@link MavenConfiguration} with the value sets.
     */
    public MavenConfiguration build() {
      return new ImmutableMavenConfiguration(localMavenRepository, remoteRepositories, ofNullable(userSettingsFile),
                                             ofNullable(globalSettingsFile), forcePolicyUpdateNever);
    }

  }

}
