/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.api;

import org.mule.maven.client.api.model.BundleDependency;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.api.model.MavenConfiguration;

import java.io.File;
import java.util.List;
import java.util.Optional;

import org.apache.maven.model.Model;

/**
 * Maven client API to work with maven artifacts.
 * <p>
 * To access get an instance of the client use {@link MavenClientProvider#discoverProvider(ClassLoader)} which will resolve the
 * implementation through SPI by loading an instance of {@link MavenClientProvider} that must be in the classpath.
 * 
 * @since 1.0
 */
public interface MavenClient {

  /**
   * @return the maven configuration of this client
   */
  MavenConfiguration getMavenConfiguration();

  /**
   * Resolves the artifact dependencies for an artifact.
   * 
   * @param artifactFile the artifact file to resolve the dependencies for. It may be an exploded archive following the mule
   *        deployable artifact structure or a compressed artifact following the maven conventions.
   * @param includeTestDependencies true if the test dependencies must be included in the list, false otherwise.
   * @param localRepositoryLocationSupplier a supplier of the local repository folder location. It may be empty in which case the
   *        one of the provided maven configuration will be used.
   * @param temporaryFolder a temporary folder that it's only mandatory if the artifact is compressed.
   * @return the list of dependencies of the artifact.
   */
  List<BundleDependency> resolveArtifactDependencies(File artifactFile,
                                                     boolean includeTestDependencies,
                                                     Optional<File> localRepositoryLocationSupplier,
                                                     Optional<File> temporaryFolder);

  /**
   * Resolves dependencies of a bundle.
   * 
   * @param includeTestDependencies true if the test dependencies must be included in the list, false otherwise.
   * @param bundleDescriptor descriptor of the artifact
   * @return the list of dependencies of the artifact
   */
  List<BundleDependency> resolveBundleDescriptorDependencies(boolean includeTestDependencies, BundleDescriptor bundleDescriptor);

  /**
   * Resolves dependencies of a bundle.
   *
   * @param includeTestDependencies true if the test dependencies must be included in the list, false otherwise.
   * @param includeProvidedDependencies true if the provided dependencies must be included in the dependency graph, false otherwise.
   * @param bundleDescriptor descriptor of the artifact
   * @return the list of dependencies of the artifact
   */
  List<BundleDependency> resolveBundleDescriptorDependencies(boolean includeTestDependencies, boolean includeProvidedDependencies,
                                                             BundleDescriptor bundleDescriptor);

  /**
   * Retrieves the {@link BundleDependency} for a {@link BundleDescriptor}.
   * 
   * @param bundleDescriptor the bundle descriptor.
   * @return the resolved {@link BundleDependency}
   */
  BundleDependency resolveBundleDescriptor(BundleDescriptor bundleDescriptor);

  /**
   * Creates the pom model by reading the pom inside the artifact. It won't solve the effective pom meaning that, for instance,
   * properties defined in parent pom will not be lookup to resolve the pom values.
   * <p>
   * The artifact can be a compressed file or an exploded artifact. The only requirement is that it contains the expected maven
   * structure.
   * 
   * @param artifactFile the compressed artifact
   * @return the effective pom model
   */
  Model getRawPomModel(File artifactFile);

  /**
   * Resolve the effective list of plugins.
   *
   * @param bundleDescriptors a list of plugins to resolve the effective list of plugins considering their dependencies
   * @return the effective list of plugins.
   */
  List<BundleDependency> resolvePluginBundleDescriptorsDependencies(List<BundleDescriptor> bundleDescriptors);
}
