/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.maven.client.api.model;

import static java.lang.String.join;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static org.mule.maven.client.api.util.Preconditions.checkState;

import java.io.File;
import java.util.List;
import java.util.Optional;
import java.util.Set;

/**
 * Immutable implementation for {@link MavenConfiguration}.
 *
 * @since 1.0
 */
class ImmutableMavenConfiguration implements MavenConfiguration {

  private final List<RemoteRepository> remoteRepositories;
  private final File localMavenRepositoryLocation;
  private final Optional<File> userSettingsFileOptional;
  private final Optional<File> globalSettingsFileOptional;
  private boolean forcePolicyUpdateNever;
  private boolean offlineMode;

  /**
   * Creates an {@link ImmutableMavenConfiguration}
   *
   * @param localMavenRepositoryLocation the maven local repository location.
   * @param remoteRepositories list of remote maven repositories to use for discovering artifacts.
   * @param userSettingsFileOptional the user settings.xml file location.
   * @param globalSettingsFileOptional the global settings.xml file location.
   * @param forcePolicyUpdateNever if true, regardless of the settings.xml configuration, for all repositories, the update policy
   *        will be configured to never update artifact.
   * @param offlineMode if true, regardless of the remote repositories configured it will only use local repository for resolutions.
   */
  public ImmutableMavenConfiguration(File localMavenRepositoryLocation, List<RemoteRepository> remoteRepositories,
                                     Optional<File> userSettingsFileOptional, Optional<File> globalSettingsFileOptional,
                                     boolean forcePolicyUpdateNever, boolean offlineMode) {
    Set<String> uniqueIdList = remoteRepositories.stream().map(RemoteRepository::getId).collect(toSet());
    checkState(uniqueIdList.size() == remoteRepositories.size(),
               "The configured set of remote repositories do not have unique IDs");
    this.localMavenRepositoryLocation = localMavenRepositoryLocation;
    this.remoteRepositories = remoteRepositories;
    this.userSettingsFileOptional = userSettingsFileOptional;
    this.globalSettingsFileOptional = globalSettingsFileOptional;
    this.forcePolicyUpdateNever = forcePolicyUpdateNever;
    this.offlineMode = offlineMode;
  }

  @Override
  public File getLocalMavenRepositoryLocation() {
    return localMavenRepositoryLocation;
  }

  @Override
  public List<RemoteRepository> getMavenRemoteRepositories() {
    return remoteRepositories;
  }

  @Override
  public Optional<File> getUserSettingsLocation() {
    return userSettingsFileOptional;
  }

  @Override
  public Optional<File> getGlobalSettingsLocation() {
    return globalSettingsFileOptional;
  }

  @Override
  public boolean getForcePolicyUpdateNever() {
    return forcePolicyUpdateNever;
  }

  @Override
  public boolean getOfflineMode() {
    return offlineMode;
  }

  @Override
  public String toString() {
    return "ImmutableMavenConfiguration{" +
        "remoteRepositories=" + repositoriesToString() +
        ", localMavenRepositoryLocation=" + localMavenRepositoryLocation +
        ", userSettingsFileOptional=" + userSettingsFileOptional +
        ", globalSettingsFileOptional=" + globalSettingsFileOptional +
        ", forcePolicyUpdateNever=" + forcePolicyUpdateNever +
        ", offlineMode=" + offlineMode +
        '}';
  }

  private String repositoriesToString() {
    return join(",\n", remoteRepositories.stream().map(RemoteRepository::toString).collect(toList()));
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    ImmutableMavenConfiguration that = (ImmutableMavenConfiguration) o;

    if (forcePolicyUpdateNever != that.forcePolicyUpdateNever) {
      return false;
    }
    if (offlineMode != that.offlineMode) {
      return false;
    }
    if (!remoteRepositories.equals(that.remoteRepositories)) {
      return false;
    }
    if (!localMavenRepositoryLocation.equals(that.localMavenRepositoryLocation)) {
      return false;
    }
    if (!userSettingsFileOptional.equals(that.userSettingsFileOptional)) {
      return false;
    }
    return globalSettingsFileOptional.equals(that.globalSettingsFileOptional);
  }

  @Override
  public int hashCode() {
    int result = remoteRepositories.hashCode();
    result = 31 * result + localMavenRepositoryLocation.hashCode();
    result = 31 * result + userSettingsFileOptional.hashCode();
    result = 31 * result + globalSettingsFileOptional.hashCode();
    result = 31 * result + (forcePolicyUpdateNever ? 1 : 0);
    result = 31 * result + (offlineMode ? 1 : 0);
    return result;
  }
}
